"""Add keyword_overrides table (Issue #449)

Revision ID: 20251228_01
Revises: 20251227_02
Create Date: 2025-12-28

Issue #449: Dynamic keyword management for NECESIDAD classifier.
Allows admins to add/edit/delete keywords from UI without code deployment.

REGLA #14: Verificación de existencia antes de crear (idempotente).
"""

import logging

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision = "20251228_01"
down_revision = "20251227_02"
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create keyword_overrides table."""
    conn = op.get_bind()

    # ====================================================================
    # STEP 1: Check if table already exists (idempotent)
    # ====================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='keyword_overrides'"
        )
    ).fetchone()

    if table_exists:
        logger.info("keyword_overrides table already exists, skipping creation")
        return

    # ====================================================================
    # STEP 2: Create table
    # ====================================================================
    op.create_table(
        "keyword_overrides",
        sa.Column("id", sa.Integer(), primary_key=True),
        sa.Column("keyword", sa.String(200), nullable=False),
        sa.Column("category", sa.String(100), nullable=False),
        sa.Column(
            "keyword_type",
            sa.String(20),
            nullable=False,
            server_default="keyword",
        ),
        sa.Column("priority", sa.Integer(), nullable=False, server_default="100"),
        sa.Column("is_active", sa.Boolean(), nullable=False, server_default="true"),
        sa.Column(
            "created_by",
            postgresql.UUID(as_uuid=True),
            sa.ForeignKey("users.id", ondelete="SET NULL"),
            nullable=True,
        ),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            server_default=sa.text("NOW()"),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            server_default=sa.text("NOW()"),
            nullable=False,
        ),
        sa.Column("notes", sa.Text(), nullable=True),
        sa.Column("source", sa.String(50), server_default="manual"),
    )
    logger.info("Created keyword_overrides table")

    # ====================================================================
    # STEP 3: Create indexes
    # ====================================================================
    # Unique constraint: one keyword per type
    op.create_unique_constraint(
        "uq_keyword_type", "keyword_overrides", ["keyword", "keyword_type"]
    )

    # Index for active keywords by type (most common query)
    op.create_index(
        "ix_keyword_override_active",
        "keyword_overrides",
        ["is_active", "keyword_type"],
    )

    # Index for category lookup
    op.create_index(
        "ix_keyword_override_category",
        "keyword_overrides",
        ["category"],
    )

    # Index on id (already auto-created by primary key)
    op.create_index(
        "ix_keyword_override_id",
        "keyword_overrides",
        ["id"],
    )

    # Case-insensitive index for keyword search
    op.execute(
        sa.text(
            """
            CREATE INDEX ix_keyword_override_keyword_lower
            ON keyword_overrides (lower(keyword))
            """
        )
    )

    logger.info("Created indexes for keyword_overrides table")


def downgrade() -> None:
    """Drop keyword_overrides table."""
    conn = op.get_bind()

    # Check if table exists before dropping
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='keyword_overrides'"
        )
    ).fetchone()

    if not table_exists:
        logger.info("keyword_overrides table does not exist, skipping drop")
        return

    # Drop indexes first
    op.drop_index("ix_keyword_override_keyword_lower", table_name="keyword_overrides")
    op.drop_index("ix_keyword_override_category", table_name="keyword_overrides")
    op.drop_index("ix_keyword_override_active", table_name="keyword_overrides")
    op.drop_index("ix_keyword_override_id", table_name="keyword_overrides")
    op.drop_constraint("uq_keyword_type", "keyword_overrides", type_="unique")

    # Drop table
    op.drop_table("keyword_overrides")
    logger.info("Dropped keyword_overrides table")
