"""Add ean_codes array to ProductCatalogVentaLibre for multi-EAN synonym support

Issue #481: Gestión de sinónimos de productos (multi-EAN/CN por producto)

Un mismo producto puede tener múltiples EANs por:
- Cambios de packaging
- Reformulaciones
- Cambios de distribuidor

Este campo permite almacenar todos los EANs válidos de un producto,
manteniendo ean13 como "EAN principal" para backwards compatibility.

Revision ID: 20251227_01
Revises: 20251226_03
Create Date: 2025-12-27

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import ARRAY

# revision identifiers, used by Alembic.
revision = '20251227_01'
down_revision = '20251226_03'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add ean_codes array field with GIN index and migrate existing data."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if 'product_catalog_venta_libre' not in tables:
        print("SKIP: Table product_catalog_venta_libre does not exist yet")
        return

    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # === 1. Añadir campo ean_codes (array de EANs) ===
    if 'ean_codes' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'ean_codes',
                ARRAY(sa.String(20)),
                nullable=True,
                server_default='{}',
                comment='Todos los EANs válidos del producto (multi-EAN synonym support)'
            )
        )
        print("  + Added column ean_codes")

        # Índice GIN para búsqueda eficiente con operador @> (contains)
        op.execute("""
            CREATE INDEX IF NOT EXISTS ix_venta_libre_ean_codes_gin
            ON product_catalog_venta_libre USING GIN (ean_codes)
        """)
        print("  + Created GIN index ix_venta_libre_ean_codes_gin")

        # === 2. Migrar ean13 existentes al array ===
        # Solo migrar si ean_codes está vacío y ean13 tiene valor
        result = op.get_bind().execute(sa.text("""
            UPDATE product_catalog_venta_libre
            SET ean_codes = ARRAY[ean13]
            WHERE ean13 IS NOT NULL
              AND ean13 != ''
              AND (ean_codes IS NULL OR ean_codes = '{}')
        """))
        print(f"  + Migrated {result.rowcount} existing ean13 values to ean_codes array")

    else:
        print("  SKIP: Column ean_codes already exists")

    print("Migration 20251227_01 completed: Multi-EAN synonym support added")


def downgrade() -> None:
    """Remove ean_codes array field."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if 'product_catalog_venta_libre' not in tables:
        return

    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # Drop index first
    op.execute("DROP INDEX IF EXISTS ix_venta_libre_ean_codes_gin")

    # Drop column
    if 'ean_codes' in existing_columns:
        op.drop_column('product_catalog_venta_libre', 'ean_codes')
        print("  - Removed column ean_codes")
