"""Add inventory file types to filetype enum

Issue #476: Inventory file upload support.

PostgreSQL enum 'filetype' needs INVENTORY_FARMANAGER and INVENTORY_FARMATIC
values to support inventory uploads from ERP systems.

Revision ID: 20251226_03
Revises: 20251226_02
Create Date: 2025-12-26

"""
import re

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251226_03'
down_revision = '20251226_02'
branch_labels = None
depends_on = None

# Valid enum value pattern: UPPERCASE letters, digits, underscores only
# Must start with a letter (defensive validation for SQL injection prevention)
VALID_ENUM_PATTERN = re.compile(r'^[A-Z][A-Z0-9_]*$')


def upgrade() -> None:
    """
    Add INVENTORY_FARMANAGER and INVENTORY_FARMATIC to filetype enum.

    REGLA #14: Idempotent migrations (verify existence BEFORE modifying).
    REGLA #7: Security first - validate enum values before SQL execution.
    """
    conn = op.get_bind()

    # List of new inventory file types to add
    # NOTE: SQLAlchemy Enum uses member NAMES by default, not VALUES
    # Existing enum values are UPPERCASE (FARMATIC, FARMANAGER, etc.)
    # So we need UPPERCASE versions for consistency
    new_values = ['INVENTORY_FARMANAGER', 'INVENTORY_FARMATIC']

    for value in new_values:
        # SECURITY: Validate enum value format before SQL execution
        # Prevents SQL injection even though values are hardcoded (defense in depth)
        if not VALID_ENUM_PATTERN.match(value):
            raise ValueError(
                f"Invalid enum value format: '{value}'. "
                f"Must match pattern: {VALID_ENUM_PATTERN.pattern}"
            )

        # Check if value already exists in enum
        result = conn.execute(sa.text(
            "SELECT EXISTS(SELECT 1 FROM pg_enum "
            "WHERE enumlabel = :value "
            "AND enumtypid = (SELECT oid FROM pg_type WHERE typname = 'filetype'))"
        ), {"value": value})

        exists = result.scalar()

        if not exists:
            # Add value to enum
            # Note: PostgreSQL doesn't support parameters for ALTER TYPE DDL
            # Value is safe because we validated the format above
            op.execute(f"ALTER TYPE filetype ADD VALUE '{value}'")
            print(f"[MIGRATION] ✓ Added '{value}' to enum filetype")
        else:
            print(f"[MIGRATION] ⊘ '{value}' already exists in enum filetype (skip)")


def downgrade() -> None:
    """
    Downgrade not implemented for safety.

    PostgreSQL doesn't support removing values from an enum directly.
    To revert, the entire enum would need to be recreated.
    """
    pass
