"""Add product_catalog_id and product_type to InventorySnapshot

Enables classification of inventory products as prescription or venta_libre,
linking to the appropriate catalog for category information.

- product_catalog_id: FK to ProductCatalog for prescription products
- product_type: 'prescription', 'venta_libre', or 'unknown'

Revision ID: 20251226_02
Revises: 20251226_01
Create Date: 2025-12-26

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID

# revision identifiers, used by Alembic.
revision = '20251226_02'
down_revision = '20251226_01'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add product_catalog_id and product_type columns."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('inventory_snapshots')]

    # Add product_catalog_id
    if 'product_catalog_id' not in existing_columns:
        op.add_column(
            'inventory_snapshots',
            sa.Column(
                'product_catalog_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='Referencia al producto en catálogo CIMA/prescripción'
            )
        )

        # Create foreign key
        op.create_foreign_key(
            'fk_inventory_product_catalog',
            'inventory_snapshots',
            'product_catalog',
            ['product_catalog_id'],
            ['id'],
            ondelete='SET NULL'
        )

        # Create index
        op.create_index(
            'ix_inventory_product_catalog',
            'inventory_snapshots',
            ['product_catalog_id'],
            unique=False,
            if_not_exists=True
        )

    # Add product_type
    if 'product_type' not in existing_columns:
        op.add_column(
            'inventory_snapshots',
            sa.Column(
                'product_type',
                sa.String(20),
                nullable=True,
                comment='Tipo: prescription, venta_libre, unknown'
            )
        )

        # Create index for filtering by type
        op.create_index(
            'ix_inventory_product_type',
            'inventory_snapshots',
            ['product_type'],
            unique=False,
            if_not_exists=True
        )


def downgrade() -> None:
    """Remove product_catalog_id and product_type columns."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('inventory_snapshots')]

    if 'product_type' in existing_columns:
        op.drop_index('ix_inventory_product_type', table_name='inventory_snapshots', if_exists=True)
        op.drop_column('inventory_snapshots', 'product_type')

    if 'product_catalog_id' in existing_columns:
        op.drop_index('ix_inventory_product_catalog', table_name='inventory_snapshots', if_exists=True)
        op.drop_constraint('fk_inventory_product_catalog', 'inventory_snapshots', type_='foreignkey')
        op.drop_column('inventory_snapshots', 'product_catalog_id')
