"""Add venta_libre_product_id to InventorySnapshot

Issue #477: Link inventory snapshots to VentaLibre catalog.

This allows tracking which products from inventory uploads
have been matched to the VentaLibre catalog, enabling
duplicate detection and catalog enrichment from inventory data.

Revision ID: 20251226_01
Revises: 20251225_01
Create Date: 2025-12-26

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID

# revision identifiers, used by Alembic.
revision = '20251226_01'
down_revision = '20251225_01'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add venta_libre_product_id column to inventory_snapshots."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('inventory_snapshots')]

    if 'venta_libre_product_id' not in existing_columns:
        op.add_column(
            'inventory_snapshots',
            sa.Column(
                'venta_libre_product_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='Referencia al producto en catálogo VentaLibre (Issue #477)'
            )
        )

        # Create foreign key
        op.create_foreign_key(
            'fk_inventory_venta_libre_product',
            'inventory_snapshots',
            'product_catalog_venta_libre',
            ['venta_libre_product_id'],
            ['id'],
            ondelete='SET NULL'
        )

        # Create index for efficient lookups
        op.create_index(
            'ix_inventory_venta_libre_product',
            'inventory_snapshots',
            ['venta_libre_product_id'],
            unique=False,
            if_not_exists=True
        )


def downgrade() -> None:
    """Remove venta_libre_product_id column from inventory_snapshots."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('inventory_snapshots')]

    if 'venta_libre_product_id' in existing_columns:
        # Drop index first
        op.drop_index('ix_inventory_venta_libre_product', table_name='inventory_snapshots', if_exists=True)

        # Drop foreign key
        op.drop_constraint('fk_inventory_venta_libre_product', 'inventory_snapshots', type_='foreignkey')

        # Drop column
        op.drop_column('inventory_snapshots', 'venta_libre_product_id')
