"""Add duplicate management fields for ProductCatalogVentaLibre

Issue #477: Admin UI para revisión de duplicados dudosos ProductCatalogVentaLibre

Añade campos para gestionar el ciclo de vida de duplicados:
- duplicate_review_status: pending_review, confirmed_different, merged
- duplicate_group_id: UUID para agrupar candidatos a duplicados
- merged_into_id: Referencia al producto primario si fue mergeado
- merged_at: Timestamp del merge
- merged_by: UUID del usuario que realizó el merge

Revision ID: 20251225_01
Revises: a09bec159d1a
Create Date: 2025-12-25

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID

# revision identifiers, used by Alembic.
revision = '20251225_01'
down_revision = 'a09bec159d1a'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add duplicate management fields to product_catalog_venta_libre."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # === 1. duplicate_review_status ===
    if 'duplicate_review_status' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'duplicate_review_status',
                sa.String(20),
                nullable=True,
                comment='Status: pending_review, confirmed_different, merged'
            )
        )
        op.create_index(
            'ix_venta_libre_duplicate_status',
            'product_catalog_venta_libre',
            ['duplicate_review_status'],
            unique=False,
            if_not_exists=True
        )

    # === 2. duplicate_group_id ===
    if 'duplicate_group_id' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'duplicate_group_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='ID del grupo de duplicados potenciales (mismo grupo = candidatos a merge)'
            )
        )
        op.create_index(
            'ix_venta_libre_duplicate_group',
            'product_catalog_venta_libre',
            ['duplicate_group_id'],
            unique=False,
            if_not_exists=True
        )

    # === 3. merged_into_id ===
    if 'merged_into_id' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'merged_into_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='ID del producto primario si este fue mergeado'
            )
        )

    # === 4. merged_at ===
    if 'merged_at' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'merged_at',
                sa.DateTime(timezone=True),
                nullable=True,
                comment='Fecha de merge'
            )
        )

    # === 5. merged_by ===
    if 'merged_by' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'merged_by',
                UUID(as_uuid=True),
                nullable=True,
                comment='UUID del usuario que realizó el merge'
            )
        )

    # === 6. Índice compuesto para consultas de admin ===
    # Buscar productos con status pending_review ordenados por similitud
    conn.execute(sa.text("""
        CREATE INDEX IF NOT EXISTS ix_venta_libre_duplicate_review_queue
        ON product_catalog_venta_libre (duplicate_review_status, duplicate_group_id)
        WHERE duplicate_review_status = 'pending_review' AND is_active = true
    """))


def downgrade() -> None:
    """Remove duplicate management fields from product_catalog_venta_libre."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # Eliminar índice parcial
    conn.execute(sa.text("""
        DROP INDEX IF EXISTS ix_venta_libre_duplicate_review_queue
    """))

    # Eliminar columnas en orden inverso
    columns_to_drop = ['merged_by', 'merged_at', 'merged_into_id', 'duplicate_group_id', 'duplicate_review_status']

    for col in columns_to_drop:
        if col in existing_columns:
            # Eliminar índices asociados primero
            if col == 'duplicate_review_status':
                op.drop_index('ix_venta_libre_duplicate_status', table_name='product_catalog_venta_libre', if_exists=True)
            elif col == 'duplicate_group_id':
                op.drop_index('ix_venta_libre_duplicate_group', table_name='product_catalog_venta_libre', if_exists=True)

            op.drop_column('product_catalog_venta_libre', col)
