"""Create inventory_snapshots table

Issue #476: Carga de ficheros de inventario desde ERPs

Crea la tabla para almacenar snapshots de inventario por farmacia y fecha.
Cada snapshot representa el estado del inventario en una fecha específica.

Revision ID: 20251224_04
Revises: 20251224_03
Create Date: 2025-12-24

"""
import logging

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID

# revision identifiers, used by Alembic.
revision = '20251224_04'
down_revision = '20251224_03'
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create inventory_snapshots table with indexes."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla ya existe (idempotente)
    tables = inspector.get_table_names()
    if 'inventory_snapshots' in tables:
        logger.info("SKIP: Table inventory_snapshots already exists")
        print("SKIP: Table inventory_snapshots already exists")
        return

    # Crear tabla
    op.create_table(
        'inventory_snapshots',
        # Primary key
        sa.Column('id', UUID(as_uuid=True), primary_key=True),

        # Foreign keys
        sa.Column(
            'pharmacy_id',
            UUID(as_uuid=True),
            sa.ForeignKey('pharmacies.id', ondelete='CASCADE'),
            nullable=False,
            index=True,
            comment='Farmacia propietaria del inventario'
        ),
        sa.Column(
            'file_upload_id',
            UUID(as_uuid=True),
            sa.ForeignKey('file_uploads.id', ondelete='CASCADE'),
            nullable=False,
            comment='Fichero de origen del snapshot'
        ),

        # Identificación del producto
        sa.Column(
            'product_code',
            sa.String(20),
            nullable=True,
            comment='Código Nacional (CN) - 6-7 dígitos'
        ),
        sa.Column(
            'product_name',
            sa.String(500),
            nullable=False,
            comment='Nombre del producto según ERP'
        ),
        sa.Column(
            'ean13',
            sa.String(20),
            nullable=True,
            comment='Código EAN-13 completo'
        ),

        # Stock
        sa.Column(
            'stock_quantity',
            sa.Integer,
            nullable=False,
            default=0,
            comment='Unidades en stock'
        ),
        sa.Column(
            'stock_value',
            sa.Numeric(12, 2),
            nullable=True,
            comment='Valor del stock a PVP (unit_price * stock_quantity)'
        ),
        sa.Column(
            'cost_value',
            sa.Numeric(12, 2),
            nullable=True,
            comment='Valor del stock a coste (unit_cost * stock_quantity)'
        ),

        # Precios unitarios
        sa.Column(
            'unit_price',
            sa.Numeric(10, 2),
            nullable=True,
            comment='Precio de venta al público (PVP)'
        ),
        sa.Column(
            'unit_cost',
            sa.Numeric(10, 2),
            nullable=True,
            comment='Precio de coste/compra (PCM)'
        ),

        # Fechas de actividad
        sa.Column(
            'last_sale_date',
            sa.Date,
            nullable=True,
            comment='Fecha de última venta'
        ),
        sa.Column(
            'last_purchase_date',
            sa.Date,
            nullable=True,
            comment='Fecha de última compra'
        ),

        # Fecha del snapshot
        sa.Column(
            'snapshot_date',
            sa.Date,
            nullable=False,
            comment='Fecha del inventario (fecha del fichero)'
        ),

        # Metadata
        sa.Column(
            'created_at',
            sa.DateTime,
            server_default=sa.func.now(),
            comment='Timestamp de creación del registro'
        ),
    )
    print("  + Created table inventory_snapshots")

    # Crear índices compuestos
    op.execute("""
        CREATE INDEX IF NOT EXISTS idx_inventory_pharmacy_date
        ON inventory_snapshots (pharmacy_id, snapshot_date)
    """)
    print("  + Created index idx_inventory_pharmacy_date")

    op.execute("""
        CREATE INDEX IF NOT EXISTS idx_inventory_product_code
        ON inventory_snapshots (product_code)
    """)
    print("  + Created index idx_inventory_product_code")

    op.execute("""
        CREATE INDEX IF NOT EXISTS idx_inventory_ean13
        ON inventory_snapshots (ean13)
    """)
    print("  + Created index idx_inventory_ean13")

    # Añadir nuevos valores al enum file_type si no existen
    # PostgreSQL requiere ALTER TYPE para añadir valores a enums
    try:
        op.execute("""
            DO $$
            BEGIN
                IF NOT EXISTS (
                    SELECT 1 FROM pg_enum
                    WHERE enumlabel = 'inventory_farmatic'
                    AND enumtypid = 'filetype'::regtype
                ) THEN
                    ALTER TYPE filetype ADD VALUE 'inventory_farmatic';
                END IF;
            EXCEPTION
                WHEN undefined_object THEN NULL;
            END $$;
        """)
        op.execute("""
            DO $$
            BEGIN
                IF NOT EXISTS (
                    SELECT 1 FROM pg_enum
                    WHERE enumlabel = 'inventory_farmanager'
                    AND enumtypid = 'filetype'::regtype
                ) THEN
                    ALTER TYPE filetype ADD VALUE 'inventory_farmanager';
                END IF;
            EXCEPTION
                WHEN undefined_object THEN NULL;
            END $$;
        """)
        print("  + Added inventory file types to filetype enum")
    except Exception as e:
        # En algunos casos el enum puede no existir o ser diferente
        logger.warning(f"Could not add enum values: {e}")
        print(f"  ! Warning: Could not add enum values: {e}")

    print("Migration 20251224_04 completed: inventory_snapshots table created")


def downgrade() -> None:
    """Drop inventory_snapshots table."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if 'inventory_snapshots' not in tables:
        return

    # Drop indexes first
    try:
        op.execute("DROP INDEX IF EXISTS idx_inventory_pharmacy_date")
        op.execute("DROP INDEX IF EXISTS idx_inventory_product_code")
        op.execute("DROP INDEX IF EXISTS idx_inventory_ean13")
    except Exception:
        pass

    # Drop table
    op.drop_table('inventory_snapshots')

    # Note: PostgreSQL does not support removing values from enums
    # The inventory_* enum values will remain (harmless)
