"""Add product_name_tokens field for fuzzy matching

Issue #473: Fuzzy matching para detectar duplicados semánticos

Añade campo product_name_tokens con tokens ordenados alfabéticamente
para detección eficiente de duplicados semánticos usando rapidfuzz.

Incluye:
- Campo product_name_tokens
- Extensión pg_trgm para búsqueda eficiente de trigramas
- Índice GIN con trigramas para queries de similitud rápidas

Ejemplo:
    "TIRAS CONTOUR NEXT 50U"   → "50u contour next tiras"
    "CONTOUR NEXT TIRAS 50 U"  → "50u contour next tiras"
    → MATCH!

Revision ID: 20251224_03
Revises: 20251224_02
Create Date: 2025-12-24

"""
from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
revision = '20251224_03'
down_revision = '20251224_02'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add product_name_tokens field with trigram index."""
    conn = op.get_bind()

    # === 1. Habilitar extensión pg_trgm para búsqueda eficiente ===
    # La extensión permite búsquedas de similitud en milisegundos
    conn.execute(sa.text("CREATE EXTENSION IF NOT EXISTS pg_trgm"))

    # === 2. Añadir columna product_name_tokens ===
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    if 'product_name_tokens' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'product_name_tokens',
                sa.String(500),
                nullable=True,
                comment='Tokens ordenados alfabéticamente para fuzzy matching'
            )
        )

    # === 3. Índice B-tree para búsqueda exacta ===
    op.create_index(
        'ix_venta_libre_name_tokens',
        'product_catalog_venta_libre',
        ['product_name_tokens'],
        unique=False,
        if_not_exists=True
    )

    # === 4. Índice GIN con trigramas para búsqueda de similitud ===
    # Esto permite queries como: similarity(tokens, 'search') > 0.4
    # Reduciendo 50,000 productos a ~20 candidatos en milisegundos
    conn.execute(sa.text("""
        CREATE INDEX IF NOT EXISTS ix_venta_libre_tokens_trgm
        ON product_catalog_venta_libre
        USING gin (product_name_tokens gin_trgm_ops)
    """))


def downgrade() -> None:
    """Remove product_name_tokens field and trigram index."""
    conn = op.get_bind()

    # Eliminar índice de trigramas
    conn.execute(sa.text("""
        DROP INDEX IF EXISTS ix_venta_libre_tokens_trgm
    """))

    # Eliminar índice B-tree
    op.drop_index(
        'ix_venta_libre_name_tokens',
        table_name='product_catalog_venta_libre',
        if_exists=True
    )

    # Verificar existencia antes de eliminar (REGLA #14: idempotente)
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    if 'product_name_tokens' in existing_columns:
        op.drop_column('product_catalog_venta_libre', 'product_name_tokens')

    # Nota: No eliminamos pg_trgm porque puede ser usada por otros índices
