"""Add cross-pharmacy reconciliation fields to ProductCatalogVentaLibre

Issue #474: Conciliación cross-farmacia por EAN/CN

Añade campos para identificación y tracking de productos entre farmacias:
- ean13: EAN-13 validado del producto
- cn_codes: Array de códigos nacionales vistos
- pharmacy_ids_seen: Array de UUIDs de farmacias
- first_pharmacy_id: Primera farmacia que reportó el producto

También añade índice GIN para búsqueda eficiente por CN.

Revision ID: 20251224_02
Revises: 20251224_01
Create Date: 2025-12-24

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID, ARRAY

# revision identifiers, used by Alembic.
revision = '20251224_02'
down_revision = '20251224_01'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add cross-pharmacy reconciliation fields."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if 'product_catalog_venta_libre' not in tables:
        print("SKIP: Table product_catalog_venta_libre does not exist yet")
        return

    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # === 1. ean13: EAN-13 principal validado ===
    if 'ean13' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'ean13',
                sa.String(13),
                nullable=True,
                comment='EAN-13 principal del producto (validado con checksum)'
            )
        )
        # Usar CREATE INDEX IF NOT EXISTS para idempotencia (REGLA #14)
        op.execute("""
            CREATE INDEX IF NOT EXISTS ix_venta_libre_ean13
            ON product_catalog_venta_libre (ean13)
        """)
        print("  + Added column ean13 with index")

    # === 2. cn_codes: Array de códigos nacionales vistos ===
    if 'cn_codes' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'cn_codes',
                ARRAY(sa.String(20)),
                nullable=True,
                server_default='{}',
                comment='Códigos nacionales vistos (6-7 dígitos, pueden variar por farmacia)'
            )
        )
        # Índice GIN para búsqueda eficiente en arrays
        op.execute("""
            CREATE INDEX IF NOT EXISTS ix_venta_libre_cn_codes_gin
            ON product_catalog_venta_libre USING GIN (cn_codes)
        """)
        print("  + Added column cn_codes with GIN index")

    # === 3. pharmacy_ids_seen: Array de UUIDs de farmacias ===
    if 'pharmacy_ids_seen' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'pharmacy_ids_seen',
                ARRAY(UUID(as_uuid=True)),
                nullable=True,
                server_default='{}',
                comment='UUIDs de farmacias que han vendido este producto'
            )
        )
        print("  + Added column pharmacy_ids_seen")

    # === 4. first_pharmacy_id: Primera farmacia que lo reportó ===
    if 'first_pharmacy_id' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'first_pharmacy_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='UUID de la primera farmacia que reportó el producto'
            )
        )
        print("  + Added column first_pharmacy_id")

    print("Migration 20251224_02 completed: Cross-pharmacy reconciliation fields added")


def downgrade() -> None:
    """Remove cross-pharmacy reconciliation fields."""

    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if 'product_catalog_venta_libre' not in tables:
        return

    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # Drop index first if exists
    try:
        op.drop_index('ix_venta_libre_ean13', table_name='product_catalog_venta_libre')
    except Exception:
        pass

    try:
        op.execute("DROP INDEX IF EXISTS ix_venta_libre_cn_codes_gin")
    except Exception:
        pass

    # Drop columns
    for col in ['first_pharmacy_id', 'pharmacy_ids_seen', 'cn_codes', 'ean13']:
        if col in existing_columns:
            op.drop_column('product_catalog_venta_libre', col)
