"""Add taxonomy labeling fields to product_cluster

Issue #462: TaxonomyLabelerService

Añade campos de control para el sistema de etiquetado:
- composition_hash: SHA256 para detectar cambios en composición
- labeled_at: Timestamp del último etiquetado
- llm_version: Versión del modelo LLM usado

Revision ID: 20251224_01
Revises: f2610595fadb
Create Date: 2024-12-24

"""
from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
revision = '20251224_01'
down_revision = 'f2610595fadb'
branch_labels = None
depends_on = None


def upgrade() -> None:
    # Verificar si las columnas ya existen antes de añadirlas
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_cluster')]

    # composition_hash: SHA256 para detectar cambios en composición del cluster
    if 'composition_hash' not in existing_columns:
        op.add_column(
            'product_cluster',
            sa.Column(
                'composition_hash',
                sa.String(64),
                nullable=True,
                comment='SHA256 hash of sorted product_ids for change detection'
            )
        )

    # labeled_at: Timestamp del último etiquetado
    if 'labeled_at' not in existing_columns:
        op.add_column(
            'product_cluster',
            sa.Column(
                'labeled_at',
                sa.DateTime(timezone=True),
                nullable=True,
                comment='Timestamp of last taxonomy labeling'
            )
        )

    # llm_version: Versión del modelo LLM usado
    if 'llm_version' not in existing_columns:
        op.add_column(
            'product_cluster',
            sa.Column(
                'llm_version',
                sa.String(50),
                nullable=True,
                comment='LLM model version used for naming (e.g., gpt-4o-mini)'
            )
        )

    # Índice para búsqueda de clusters sin etiquetar o stale
    op.create_index(
        'ix_product_cluster_labeled_at',
        'product_cluster',
        ['labeled_at'],
        unique=False,
        if_not_exists=True
    )


def downgrade() -> None:
    # Eliminar índice
    op.drop_index('ix_product_cluster_labeled_at', table_name='product_cluster', if_exists=True)

    # Verificar existencia antes de eliminar columnas (REGLA #14: idempotente)
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_cluster')]

    if 'llm_version' in existing_columns:
        op.drop_column('product_cluster', 'llm_version')
    if 'labeled_at' in existing_columns:
        op.drop_column('product_cluster', 'labeled_at')
    if 'composition_hash' in existing_columns:
        op.drop_column('product_cluster', 'composition_hash')
