"""Add UMAP coordinates to ProductCatalogVentaLibre

Revision ID: 20251220_03
Revises: 20251220_02_rename_tos
Create Date: 2025-12-20

Issue #458: Add umap_x, umap_y, umap_version columns for UMAP 2D visualization.
These coordinates are computed by UMAPModelService and used by the frontend
to display products in a 2D scatter plot.

Columns:
- umap_x: Float, X coordinate in UMAP 2D space
- umap_y: Float, Y coordinate in UMAP 2D space
- umap_version: String, model version (e.g., "v1.0") for tracking retrains
"""

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251220_03_add_umap'
down_revision = '20251220_02_rename_tos'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add UMAP coordinate columns to product_catalog_venta_libre."""
    conn = op.get_bind()

    # REGLA #14: Verificar existencia antes de añadir columnas (idempotente)

    # Check if umap_x already exists
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_catalog_venta_libre'
        AND column_name = 'umap_x'
    """))
    if not result.fetchone():
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column('umap_x', sa.Float, nullable=True,
                      comment='X coordinate in UMAP 2D space')
        )
        print("Added column: umap_x")

    # Check if umap_y already exists
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_catalog_venta_libre'
        AND column_name = 'umap_y'
    """))
    if not result.fetchone():
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column('umap_y', sa.Float, nullable=True,
                      comment='Y coordinate in UMAP 2D space')
        )
        print("Added column: umap_y")

    # Check if umap_version already exists
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_catalog_venta_libre'
        AND column_name = 'umap_version'
    """))
    if not result.fetchone():
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column('umap_version', sa.String(20), nullable=True,
                      comment='UMAP model version (e.g., v1.0)')
        )
        print("Added column: umap_version")

    # Create composite index for efficient UMAP data queries
    # Index on (umap_version, umap_x, umap_y) for filtering by version
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE tablename = 'product_catalog_venta_libre'
        AND indexname = 'ix_venta_libre_umap_coords'
    """))
    if not result.fetchone():
        op.create_index(
            'ix_venta_libre_umap_coords',
            'product_catalog_venta_libre',
            ['umap_version'],
            postgresql_where=sa.text('umap_x IS NOT NULL')
        )
        print("Created index: ix_venta_libre_umap_coords")


def downgrade() -> None:
    """Remove UMAP coordinate columns."""
    conn = op.get_bind()

    # Drop index first
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE tablename = 'product_catalog_venta_libre'
        AND indexname = 'ix_venta_libre_umap_coords'
    """))
    if result.fetchone():
        op.drop_index('ix_venta_libre_umap_coords', 'product_catalog_venta_libre')

    # Drop columns (check existence for idempotency)
    # Using parameterized queries with bindparam for column name checks
    for col in ['umap_version', 'umap_y', 'umap_x']:
        result = conn.execute(
            sa.text("""
                SELECT column_name FROM information_schema.columns
                WHERE table_name = 'product_catalog_venta_libre'
                AND column_name = :col_name
            """),
            {"col_name": col}
        )
        if result.fetchone():
            op.drop_column('product_catalog_venta_libre', col)
            print(f"Dropped column: {col}")
