"""Create ProductCatalogVentaLibre table (Issue #457)

Revision ID: 20251220_01_venta_libre_catalog
Revises: 20251219_02_feedback_idx
Create Date: 2025-12-20

Issue #457 M4 Feedback Loop - Catálogo interno de productos venta libre.

Este catálogo es el equivalente a ProductCatalog (CIMA) pero para productos
que NO tienen código nacional. Se puebla automáticamente desde las ventas
de todas las farmacias y se enriquece con clasificación ML + validación humana.

Arquitectura:
    ProductCatalog (CIMA)           → Medicamentos con CN (fuente externa)
    ProductCatalogVet (CIMAVET)     → Veterinaria con CN (fuente externa)
    ProductCatalogVentaLibre        → Parafarmacia sin CN (fuente INTERNA)

Flujo de enriquecimiento:
    1. Venta llega (product_name del ERP)
    2. Si tiene CN → buscar en ProductCatalog/ProductCatalogVet
    3. Si NO tiene CN → buscar en ProductCatalogVentaLibre
       - Si existe → usar clasificación cacheada
       - Si no existe → clasificar + crear entrada
    4. SalesEnrichment referencia el catálogo correspondiente

Beneficios:
    - Eliminación de duplicados: Cada producto único tiene UNA entrada
    - Performance: Clasificación cacheada por producto, no por venta
    - Consistencia: Correcciones humanas se aplican a TODAS las ventas
    - Escalabilidad: Cola de revisión con productos únicos
"""

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID, ARRAY


# revision identifiers, used by Alembic.
revision = '20251220_01_venta_libre_catalog'
down_revision = '20251219_02_feedback_idx'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Create ProductCatalogVentaLibre table and FK in SalesEnrichment."""

    conn = op.get_bind()

    # === 1. Create product_catalog_venta_libre table ===
    result = conn.execute(sa.text("""
        SELECT tablename FROM pg_tables
        WHERE tablename = 'product_catalog_venta_libre'
    """))
    if not result.fetchone():
        op.create_table(
            'product_catalog_venta_libre',

            # === IDENTIFICACIÓN ===
            sa.Column('id', UUID(as_uuid=True), primary_key=True,
                      server_default=sa.text('gen_random_uuid()')),

            # Nombre normalizado (lowercase, sin espacios extra) - CLAVE ÚNICA
            sa.Column('product_name_normalized', sa.String(500), unique=True,
                      nullable=False, index=True,
                      comment='Nombre normalizado para matching (lowercase, trim)'),

            # Nombre para mostrar (primer nombre visto, más legible)
            sa.Column('product_name_display', sa.String(500), nullable=False,
                      comment='Nombre original para UI (primer nombre visto)'),

            # Variantes del nombre vistas (para fuzzy matching futuro)
            sa.Column('product_name_variants', ARRAY(sa.String(500)), nullable=True,
                      comment='Lista de variantes del nombre vistas en distintas farmacias'),

            # === CLASIFICACIÓN ML (equivalente a ATC en CIMA) ===
            sa.Column('ml_category', sa.String(100), nullable=True, index=True,
                      comment='Categoría predicha (necesidad_especifica de symptom_taxonomy)'),

            sa.Column('ml_confidence', sa.Float(), nullable=True,
                      comment='Confianza de la predicción (0.0 - 1.0)'),

            sa.Column('prediction_source', sa.String(50), nullable=True,
                      comment='Fuente: tier1_specific, tier1_generic, brand, llm, human'),

            sa.Column('detected_brand', sa.String(100), nullable=True, index=True,
                      comment='Marca detectada (ISDIN, CERAVE, etc.)'),

            # === VALIDACIÓN HUMANA ===
            sa.Column('human_verified', sa.Boolean(), default=False, index=True,
                      comment='True si un humano ha validado/corregido la clasificación'),

            sa.Column('verified_category', sa.String(100), nullable=True,
                      comment='Categoría final tras validación humana (puede diferir de ml_category)'),

            sa.Column('verified_at', sa.DateTime(timezone=True), nullable=True,
                      comment='Fecha de verificación humana'),

            sa.Column('reviewer_notes', sa.Text(), nullable=True,
                      comment='Notas del revisor durante validación'),

            # === ESTADÍSTICAS DE USO ===
            sa.Column('first_seen_at', sa.DateTime(timezone=True),
                      server_default=sa.text('NOW()'),
                      comment='Primera vez que se vio este producto'),

            sa.Column('last_seen_at', sa.DateTime(timezone=True), nullable=True,
                      comment='Última vez que se vio este producto en una venta'),

            sa.Column('total_sales_count', sa.Integer(), default=0,
                      comment='Número total de ventas de este producto (todas las farmacias)'),

            sa.Column('pharmacies_count', sa.Integer(), default=0,
                      comment='En cuántas farmacias distintas se ha vendido'),

            # === METADATA ===
            sa.Column('is_active', sa.Boolean(), default=True,
                      comment='False para productos descatalogados o erróneos'),

            sa.Column('is_outlier', sa.Boolean(), default=False,
                      comment='True si es pack/descatalogado/ambiguo (no clasificable)'),

            sa.Column('outlier_reason', sa.String(50), nullable=True,
                      comment='Razón de outlier: pack_promocional, descatalogado, ambiguo, etc.'),

            # === TIMESTAMPS ===
            sa.Column('created_at', sa.DateTime(timezone=True),
                      server_default=sa.text('NOW()')),

            sa.Column('updated_at', sa.DateTime(timezone=True), nullable=True),
        )

    # === 2. Create indexes for product_catalog_venta_libre ===

    # Index: Feedback queue (category + not verified + active)
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'ix_venta_libre_feedback_queue'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX ix_venta_libre_feedback_queue
            ON product_catalog_venta_libre (ml_category, human_verified, is_active)
        """)

    # Index: Brand lookup
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'ix_venta_libre_brand'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX ix_venta_libre_brand
            ON product_catalog_venta_libre (detected_brand)
        """)

    # Index: Sales count (for top products)
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'ix_venta_libre_sales_count'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX ix_venta_libre_sales_count
            ON product_catalog_venta_libre (total_sales_count DESC)
        """)

    # === 3. Add FK column to sales_enrichment ===
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'sales_enrichment'
        AND column_name = 'venta_libre_product_id'
    """))
    if not result.fetchone():
        op.add_column(
            'sales_enrichment',
            sa.Column(
                'venta_libre_product_id',
                UUID(as_uuid=True),
                nullable=True,
                comment='FK a catálogo interno de venta libre (productos sin CN)'
            )
        )

        # Add FK constraint
        op.create_foreign_key(
            'fk_sales_enrichment_venta_libre_product',
            'sales_enrichment',
            'product_catalog_venta_libre',
            ['venta_libre_product_id'],
            ['id']
        )

        # Add index for FK
        op.create_index(
            'ix_sales_enrichment_venta_libre_product_id',
            'sales_enrichment',
            ['venta_libre_product_id']
        )


def downgrade() -> None:
    """Remove ProductCatalogVentaLibre and FK from SalesEnrichment."""

    conn = op.get_bind()

    # === 1. Remove FK and column from sales_enrichment ===
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'sales_enrichment'
        AND column_name = 'venta_libre_product_id'
    """))
    if result.fetchone():
        # Drop index first
        op.execute("DROP INDEX IF EXISTS ix_sales_enrichment_venta_libre_product_id")
        # Drop FK constraint
        op.drop_constraint(
            'fk_sales_enrichment_venta_libre_product',
            'sales_enrichment',
            type_='foreignkey'
        )
        # Drop column
        op.drop_column('sales_enrichment', 'venta_libre_product_id')

    # === 2. Drop indexes ===
    op.execute("DROP INDEX IF EXISTS ix_venta_libre_sales_count")
    op.execute("DROP INDEX IF EXISTS ix_venta_libre_brand")
    op.execute("DROP INDEX IF EXISTS ix_venta_libre_feedback_queue")

    # === 3. Drop table ===
    op.drop_table('product_catalog_venta_libre')
