"""Add performance indexes for FeedbackService (Issue #457)

Revision ID: 20251219_02_feedback_idx
Revises: 20251219_01_outlier
Create Date: 2025-12-19

Performance optimization for FeedbackService:
- idx_feedback_queue_priority: Composite index for review queue queries
  (product_type, ml_category, ml_confidence, created_at DESC)
  with partial filter WHERE product_type = 'venta_libre'

- idx_corrections_enrichment_lookup: Index for NOT EXISTS subquery
  optimization on product_corrections.sales_enrichment_id

- idx_corrections_type_pharmacy: Index for get_corrections_stats grouping

Expected impact:
- ~60-70% reduction in query execution time for get_review_queue
- ~50% reduction in memory usage
- Prevents full table scans on product_corrections
"""

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251219_02_feedback_idx'
down_revision = '20251219_01_outlier'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add performance indexes for FeedbackService."""

    conn = op.get_bind()

    # Index 1: Composite index for feedback queue priority ordering
    # Partial index only for venta_libre products (reduces index size significantly)
    # Note: Using regular CREATE INDEX (not CONCURRENTLY) for Alembic transaction compatibility
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'idx_feedback_queue_priority'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX idx_feedback_queue_priority
            ON sales_enrichment (product_type, ml_category, ml_confidence, created_at DESC)
            WHERE product_type = 'venta_libre'
        """)

    # Index 2: Index for NOT EXISTS subquery on product_corrections
    # Critical for avoiding full table scans in exclusion queries
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'idx_corrections_enrichment_lookup'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX idx_corrections_enrichment_lookup
            ON product_corrections (sales_enrichment_id)
        """)

    # Index 3: Index for get_corrections_stats grouping
    # Optimizes COUNT GROUP BY correction_type, pharmacy_id
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE indexname = 'idx_corrections_type_pharmacy'
    """))
    if not result.fetchone():
        op.execute("""
            CREATE INDEX idx_corrections_type_pharmacy
            ON product_corrections (correction_type, pharmacy_id)
        """)


def downgrade() -> None:
    """Remove performance indexes."""
    op.execute("DROP INDEX IF EXISTS idx_corrections_type_pharmacy")
    op.execute("DROP INDEX IF EXISTS idx_corrections_enrichment_lookup")
    op.execute("DROP INDEX IF EXISTS idx_feedback_queue_priority")
