"""Add outlier investigation fields (Issue #457)

Revision ID: 20251219_01_outlier
Revises: d9e81c815291
Create Date: 2025-12-19

Issue #457: M4 Feedback Loop - Outlier Investigation System

Adds fields to track outlier reasons and status for later investigation:
- outlier_reason: Why the product was marked as outlier
- outlier_status: Current investigation status
- outlier_resolved_at: When the outlier was resolved
- outlier_resolution_notes: Notes from resolution

This prevents outliers from becoming "orphaned garbage" in the database.
"""

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251219_01_outlier'
down_revision = 'd9e81c815291'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add outlier investigation fields to product_corrections table."""

    # Check if columns already exist before adding (idempotent)
    conn = op.get_bind()

    # Check outlier_reason column
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_corrections' AND column_name = 'outlier_reason'
    """))
    if not result.fetchone():
        # Create enum type first
        op.execute("""
            DO $$ BEGIN
                CREATE TYPE outlier_reason_enum AS ENUM (
                    'pack_promocional',
                    'descatalogado',
                    'ambiguo',
                    'nueva_categoria',
                    'otro'
                );
            EXCEPTION
                WHEN duplicate_object THEN null;
            END $$;
        """)

        op.add_column(
            'product_corrections',
            sa.Column(
                'outlier_reason',
                sa.Enum(
                    'pack_promocional', 'descatalogado', 'ambiguo',
                    'nueva_categoria', 'otro',
                    name='outlier_reason_enum'
                ),
                nullable=True
            )
        )

    # Check outlier_status column
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_corrections' AND column_name = 'outlier_status'
    """))
    if not result.fetchone():
        # Create enum type first
        op.execute("""
            DO $$ BEGIN
                CREATE TYPE outlier_status_enum AS ENUM (
                    'pending_review',
                    'investigating',
                    'resolved_category',
                    'resolved_corrected',
                    'true_outlier'
                );
            EXCEPTION
                WHEN duplicate_object THEN null;
            END $$;
        """)

        op.add_column(
            'product_corrections',
            sa.Column(
                'outlier_status',
                sa.Enum(
                    'pending_review', 'investigating', 'resolved_category',
                    'resolved_corrected', 'true_outlier',
                    name='outlier_status_enum'
                ),
                nullable=True
            )
        )

    # Check outlier_resolved_at column
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_corrections' AND column_name = 'outlier_resolved_at'
    """))
    if not result.fetchone():
        op.add_column(
            'product_corrections',
            sa.Column('outlier_resolved_at', sa.DateTime(timezone=True), nullable=True)
        )

    # Check outlier_resolution_notes column
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_corrections' AND column_name = 'outlier_resolution_notes'
    """))
    if not result.fetchone():
        op.add_column(
            'product_corrections',
            sa.Column('outlier_resolution_notes', sa.Text(), nullable=True)
        )

    # Update existing outliers to have pending_review status
    op.execute("""
        UPDATE product_corrections
        SET outlier_status = 'pending_review',
            outlier_reason = 'otro'
        WHERE correction_type = 'outlier'
        AND outlier_status IS NULL
    """)


def downgrade() -> None:
    """Remove outlier investigation fields."""
    op.drop_column('product_corrections', 'outlier_resolution_notes')
    op.drop_column('product_corrections', 'outlier_resolved_at')
    op.drop_column('product_corrections', 'outlier_status')
    op.drop_column('product_corrections', 'outlier_reason')

    # Drop enum types
    op.execute("DROP TYPE IF EXISTS outlier_status_enum")
    op.execute("DROP TYPE IF EXISTS outlier_reason_enum")
