"""Add detected_brand and brand_line columns to product_catalog

Issue #446: Store brand information in catalog for future analyses

Revision ID: 20251216_02_brand_catalog
Revises: 20251216_01_venta_libre
Create Date: 2025-12-16
"""
from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251216_02_brand_catalog'
down_revision = '20251216_01_venta_libre'
branch_labels = None
depends_on = None


def upgrade():
    """Add brand columns to product_catalog for venta_libre products analysis."""
    conn = op.get_bind()

    # Check if detected_brand column exists
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_catalog' AND column_name = 'detected_brand'
    """))
    if not result.fetchone():
        op.add_column('product_catalog',
            sa.Column('detected_brand', sa.String(100), nullable=True,
                      comment='Marca detectada por BrandDetectionService')
        )

    # Check if index exists before creating
    result = conn.execute(sa.text("""
        SELECT indexname FROM pg_indexes
        WHERE tablename = 'product_catalog' AND indexname = 'ix_product_catalog_detected_brand'
    """))
    if not result.fetchone():
        op.create_index('ix_product_catalog_detected_brand', 'product_catalog', ['detected_brand'])

    # Check if brand_line column exists
    result = conn.execute(sa.text("""
        SELECT column_name FROM information_schema.columns
        WHERE table_name = 'product_catalog' AND column_name = 'brand_line'
    """))
    if not result.fetchone():
        op.add_column('product_catalog',
            sa.Column('brand_line', sa.String(100), nullable=True,
                      comment='Línea de producto dentro de la marca (ej: FOTOPROTECTOR para ISDIN)')
        )


def downgrade():
    """Remove brand columns from product_catalog."""
    op.drop_index('ix_product_catalog_detected_brand', table_name='product_catalog')
    op.drop_column('product_catalog', 'detected_brand')
    op.drop_column('product_catalog', 'brand_line')
