"""Add venta libre taxonomy hierarchy (Issue #446)

Revision ID: 20251216_01_venta_libre
Revises: 20251210_subscription_exp
Create Date: 2025-12-16

Issue #446: Taxonomía jerárquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Cambios:
1. Nueva tabla: intercambiable_group (grupos de productos sustituibles)
2. Nuevos campos en sales_enrichment:
   - ml_subcategory (nivel 2 de jerarquía)
   - detected_brand (marca detectada)
   - brand_line (línea de producto)
   - intercambiable_group_id (FK a grupos intercambiables)
"""

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql


# revision identifiers, used by Alembic.
revision = "20251216_01_venta_libre"
down_revision = "20251210_subscription_exp"
branch_labels = None
depends_on = None


def upgrade() -> None:
    """
    Migración idempotente: verifica existencia antes de crear.
    Siguiendo REGLA #14 de CLAUDE.md.
    """
    conn = op.get_bind()

    # === 1. CREAR TABLA intercambiable_group ===
    # Verificar si la tabla ya existe
    table_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.tables
                WHERE table_schema = 'public'
                AND table_name = 'intercambiable_group'
            )
        """)
    ).scalar()

    if not table_exists:
        op.create_table(
            'intercambiable_group',
            # Clave primaria
            sa.Column('id', postgresql.UUID(as_uuid=True), nullable=False),

            # Identificación
            sa.Column('name', sa.String(200), nullable=False,
                      comment="Nombre descriptivo del grupo"),
            sa.Column('slug', sa.String(100), nullable=False,
                      comment="Identificador URL-friendly único"),
            sa.Column('description', sa.Text(), nullable=True,
                      comment="Descripción detallada del grupo"),

            # Clasificación (Niveles 1-2)
            sa.Column('necesidad', sa.String(100), nullable=False,
                      comment="NECESIDAD principal (nivel 1)"),
            sa.Column('subcategory', sa.String(100), nullable=True,
                      comment="Subcategoría (nivel 2)"),

            # Criterios de formación
            sa.Column('min_brands', sa.Integer(), nullable=False, server_default='2',
                      comment="Mínimo de marcas diferentes"),
            sa.Column('min_products', sa.Integer(), nullable=False, server_default='3',
                      comment="Mínimo de productos"),
            sa.Column('clustering_method', sa.String(50), nullable=True,
                      comment="Método: semantic, rule, manual"),
            sa.Column('clustering_params', sa.Text(), nullable=True,
                      comment="Parámetros del clustering (JSON)"),

            # Validación
            sa.Column('validated', sa.Boolean(), nullable=False, server_default='false',
                      comment="Validado por farmacéutico"),
            sa.Column('validated_by', sa.String(100), nullable=True),
            sa.Column('validated_at', sa.DateTime(timezone=True), nullable=True),
            sa.Column('validation_notes', sa.Text(), nullable=True),

            # Estadísticas denormalizadas
            sa.Column('product_count', sa.Integer(), nullable=False, server_default='0'),
            sa.Column('brand_count', sa.Integer(), nullable=False, server_default='0'),
            sa.Column('total_sales_amount', sa.Numeric(12, 2), nullable=False, server_default='0'),
            sa.Column('total_sales_count', sa.Integer(), nullable=False, server_default='0'),

            # Timestamps
            sa.Column('created_at', sa.DateTime(timezone=True),
                      server_default=sa.text('now()'), nullable=False),
            sa.Column('updated_at', sa.DateTime(timezone=True), nullable=True),

            # Constraints
            sa.PrimaryKeyConstraint('id'),
            sa.UniqueConstraint('slug', name='uq_intercambiable_group_slug'),
        )

        # Índices
        op.create_index('ix_intercambiable_group_necesidad', 'intercambiable_group', ['necesidad'])
        op.create_index('ix_intercambiable_group_slug', 'intercambiable_group', ['slug'])
        op.create_index('ix_intercambiable_group_validated', 'intercambiable_group', ['validated'])

    # === 2. AÑADIR CAMPOS A sales_enrichment ===

    # 2.1 ml_subcategory
    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_schema = 'public'
                AND table_name = 'sales_enrichment'
                AND column_name = 'ml_subcategory'
            )
        """)
    ).scalar()

    if not column_exists:
        op.add_column('sales_enrichment', sa.Column(
            'ml_subcategory', sa.String(100), nullable=True,
            comment="Subcategoría NECESIDAD nivel 2"
        ))
        op.create_index(
            'ix_sales_enrichment_ml_subcategory',
            'sales_enrichment',
            ['ml_subcategory']
        )

    # 2.2 detected_brand
    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_schema = 'public'
                AND table_name = 'sales_enrichment'
                AND column_name = 'detected_brand'
            )
        """)
    ).scalar()

    if not column_exists:
        op.add_column('sales_enrichment', sa.Column(
            'detected_brand', sa.String(100), nullable=True,
            comment="Marca detectada del producto"
        ))
        op.create_index(
            'ix_sales_enrichment_detected_brand',
            'sales_enrichment',
            ['detected_brand']
        )

    # 2.3 brand_line
    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_schema = 'public'
                AND table_name = 'sales_enrichment'
                AND column_name = 'brand_line'
            )
        """)
    ).scalar()

    if not column_exists:
        op.add_column('sales_enrichment', sa.Column(
            'brand_line', sa.String(200), nullable=True,
            comment="Línea de producto dentro de la marca"
        ))

    # 2.4 intercambiable_group_id
    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_schema = 'public'
                AND table_name = 'sales_enrichment'
                AND column_name = 'intercambiable_group_id'
            )
        """)
    ).scalar()

    if not column_exists:
        op.add_column('sales_enrichment', sa.Column(
            'intercambiable_group_id',
            postgresql.UUID(as_uuid=True),
            nullable=True,
            comment="FK a grupo de productos intercambiables"
        ))
        op.create_index(
            'ix_sales_enrichment_intercambiable_group_id',
            'sales_enrichment',
            ['intercambiable_group_id']
        )
        op.create_foreign_key(
            'fk_sales_enrichment_intercambiable_group',
            'sales_enrichment',
            'intercambiable_group',
            ['intercambiable_group_id'],
            ['id'],
            ondelete='SET NULL'
        )

    # === 3. ÍNDICE COMPUESTO PARA ANÁLISIS ===
    # Índice para queries de análisis por necesidad + subcategoría + marca
    index_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM pg_indexes
                WHERE schemaname = 'public'
                AND tablename = 'sales_enrichment'
                AND indexname = 'ix_sales_enrichment_taxonomy_hierarchy'
            )
        """)
    ).scalar()

    if not index_exists:
        op.create_index(
            'ix_sales_enrichment_taxonomy_hierarchy',
            'sales_enrichment',
            ['ml_category', 'ml_subcategory', 'detected_brand']
        )


def downgrade() -> None:
    """
    Rollback: eliminar campos y tabla en orden inverso.
    """
    conn = op.get_bind()

    # === 1. ELIMINAR ÍNDICE COMPUESTO ===
    index_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM pg_indexes
                WHERE schemaname = 'public'
                AND tablename = 'sales_enrichment'
                AND indexname = 'ix_sales_enrichment_taxonomy_hierarchy'
            )
        """)
    ).scalar()

    if index_exists:
        op.drop_index('ix_sales_enrichment_taxonomy_hierarchy', table_name='sales_enrichment')

    # === 2. ELIMINAR CAMPOS DE sales_enrichment ===

    # 2.1 intercambiable_group_id (FK primero)
    fk_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.table_constraints
                WHERE constraint_name = 'fk_sales_enrichment_intercambiable_group'
                AND table_name = 'sales_enrichment'
            )
        """)
    ).scalar()

    if fk_exists:
        op.drop_constraint('fk_sales_enrichment_intercambiable_group', 'sales_enrichment', type_='foreignkey')

    index_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM pg_indexes
                WHERE indexname = 'ix_sales_enrichment_intercambiable_group_id'
            )
        """)
    ).scalar()

    if index_exists:
        op.drop_index('ix_sales_enrichment_intercambiable_group_id', table_name='sales_enrichment')

    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_name = 'sales_enrichment'
                AND column_name = 'intercambiable_group_id'
            )
        """)
    ).scalar()

    if column_exists:
        op.drop_column('sales_enrichment', 'intercambiable_group_id')

    # 2.2 brand_line
    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_name = 'sales_enrichment'
                AND column_name = 'brand_line'
            )
        """)
    ).scalar()

    if column_exists:
        op.drop_column('sales_enrichment', 'brand_line')

    # 2.3 detected_brand
    index_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM pg_indexes
                WHERE indexname = 'ix_sales_enrichment_detected_brand'
            )
        """)
    ).scalar()

    if index_exists:
        op.drop_index('ix_sales_enrichment_detected_brand', table_name='sales_enrichment')

    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_name = 'sales_enrichment'
                AND column_name = 'detected_brand'
            )
        """)
    ).scalar()

    if column_exists:
        op.drop_column('sales_enrichment', 'detected_brand')

    # 2.4 ml_subcategory
    index_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM pg_indexes
                WHERE indexname = 'ix_sales_enrichment_ml_subcategory'
            )
        """)
    ).scalar()

    if index_exists:
        op.drop_index('ix_sales_enrichment_ml_subcategory', table_name='sales_enrichment')

    column_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.columns
                WHERE table_name = 'sales_enrichment'
                AND column_name = 'ml_subcategory'
            )
        """)
    ).scalar()

    if column_exists:
        op.drop_column('sales_enrichment', 'ml_subcategory')

    # === 3. ELIMINAR TABLA intercambiable_group ===
    table_exists = conn.execute(
        sa.text("""
            SELECT EXISTS (
                SELECT FROM information_schema.tables
                WHERE table_name = 'intercambiable_group'
            )
        """)
    ).scalar()

    if table_exists:
        op.drop_index('ix_intercambiable_group_validated', table_name='intercambiable_group')
        op.drop_index('ix_intercambiable_group_slug', table_name='intercambiable_group')
        op.drop_index('ix_intercambiable_group_necesidad', table_name='intercambiable_group')
        op.drop_table('intercambiable_group')
