"""add_cima_uppercase_to_synctype_enum

Revision ID: 20251205_add_cima_uppercase
Revises: 20251203_02_increase_nregistro
Create Date: 2025-12-05

PROBLEMA:
- El enum synctype fue creado con 'cima' (minúscula)
- El modelo Python SyncType.CIMA usa 'CIMA' (MAYÚSCULA)
- PostgreSQL es case-sensitive para enums
- Error: invalid input value for enum synctype: "CIMA"

SOLUCIÓN:
- Añadir 'CIMA' (MAYÚSCULA) al enum synctype
- Mantener 'cima' (minúscula) por compatibilidad con datos existentes

REFERENCIA:
- REGLA #14: Migraciones idempotentes (verificar existencia ANTES)
- Issue: CIMA sync failing in production
"""

from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


revision: str = "20251205_add_cima_uppercase"
down_revision: Union[str, None] = "20251203_02_increase_nregistro"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Agrega 'CIMA' (MAYÚSCULA) al enum synctype de forma idempotente.
    """
    conn = op.get_bind()

    # Verificar si el valor 'CIMA' ya existe en el enum synctype
    result = conn.execute(
        sa.text(
            "SELECT EXISTS(SELECT 1 FROM pg_enum "
            "WHERE enumlabel = 'CIMA' "
            "AND enumtypid = (SELECT oid FROM pg_type WHERE typname = 'synctype'))"
        )
    )

    cima_exists = result.scalar()

    if not cima_exists:
        # Agregar valor 'CIMA' al enum synctype
        op.execute("ALTER TYPE synctype ADD VALUE 'CIMA'")
        print("[MIGRATION] ✓ Agregado 'CIMA' a enum synctype")
    else:
        print("[MIGRATION] ⊘ 'CIMA' ya existe en enum synctype (skip)")


def downgrade() -> None:
    """
    PostgreSQL NO permite eliminar valores de un enum directamente.
    Esta migración NO es reversible automáticamente.
    """
    # No implementado por seguridad - eliminar valores de enum requiere
    # recrear el tipo completo, lo cual puede romper datos existentes
    pass
