"""add_cima_nregistro_to_product_catalog

Revision ID: 20251203_add_cima_nregistro
Revises: 20251129_add_no_comercializado
Create Date: 2025-12-03

Añade campo cima_nregistro a product_catalog para almacenar el número
de registro del medicamento desde CIMA. Este campo es necesario para
el backfill de códigos ATC.

Issue: Investigación ATC - El nregistro no se puede deducir del código nacional.

Migración 100% idempotente (verifica existencia antes de modificar)
"""

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = "20251203_add_cima_nregistro"
down_revision = "20251129_add_no_comercializado"
branch_labels = None
depends_on = None


def column_exists(table_name: str, column_name: str) -> bool:
    """Verifica si una columna existe en una tabla."""
    conn = op.get_bind()
    result = conn.execute(
        sa.text(
            """
            SELECT column_name
            FROM information_schema.columns
            WHERE table_name = :table_name
            AND column_name = :column_name
            """
        ),
        {"table_name": table_name, "column_name": column_name},
    )
    return result.fetchone() is not None


def index_exists(index_name: str) -> bool:
    """Verifica si un índice existe."""
    conn = op.get_bind()
    result = conn.execute(
        sa.text(
            """
            SELECT indexname
            FROM pg_indexes
            WHERE indexname = :index_name
            """
        ),
        {"index_name": index_name},
    )
    return result.fetchone() is not None


def upgrade() -> None:
    # Añadir columna cima_nregistro si no existe
    if not column_exists("product_catalog", "cima_nregistro"):
        op.add_column(
            "product_catalog",
            sa.Column("cima_nregistro", sa.String(20), nullable=True),
        )

    # Crear índice para búsquedas por nregistro si no existe
    if not index_exists("ix_product_catalog_cima_nregistro"):
        op.create_index(
            "ix_product_catalog_cima_nregistro",
            "product_catalog",
            ["cima_nregistro"],
            unique=False,
        )


def downgrade() -> None:
    # Eliminar índice si existe
    if index_exists("ix_product_catalog_cima_nregistro"):
        op.drop_index("ix_product_catalog_cima_nregistro", table_name="product_catalog")

    # Eliminar columna si existe
    if column_exists("product_catalog", "cima_nregistro"):
        op.drop_column("product_catalog", "cima_nregistro")
