"""add_no_comercializado_to_prescription_category_enum

Revision ID: 20251129_add_no_comercializado
Revises: 3eb9d3466183
Create Date: 2025-11-29

Issue #432: Añadir categoría NO_COMERCIALIZADO para productos de baja

Esta migración añade un nuevo valor al enum prescription_category_enum:
- 'no_comercializado': Productos con cima_estado_registro = "NO_COMERCIALIZADO"

Separar estos productos de USO_HUMANO_NO_FINANCIADO permite mejor análisis
de productos activos vs. productos retirados del mercado.

Migración 100% idempotente (verifica existencia antes de modificar)
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '20251129_add_no_comercializado'
down_revision: Union[str, None] = '3eb9d3466183'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Añade valor 'no_comercializado' al enum prescription_category_enum

    PostgreSQL permite añadir valores a un ENUM existente con ALTER TYPE.
    Verificamos existencia antes de añadir para ser idempotentes.
    """
    conn = op.get_bind()

    # === PASO 1: Verificar que el tipo ENUM existe ===
    enum_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1 FROM pg_type
            WHERE typname = 'prescription_category_enum'
        )
    """)).scalar()

    if not enum_exists:
        print("⚠️  Tipo ENUM 'prescription_category_enum' no existe - saltando migración")
        return

    # === PASO 2: Verificar si el valor ya existe en el ENUM ===
    value_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1 FROM pg_enum
            WHERE enumtypid = (SELECT oid FROM pg_type WHERE typname = 'prescription_category_enum')
              AND enumlabel = 'no_comercializado'
        )
    """)).scalar()

    if value_exists:
        print("⏭️  Valor 'no_comercializado' ya existe en el ENUM (skip)")
        return

    # === PASO 3: Añadir nuevo valor al ENUM ===
    # IMPORTANTE: En PostgreSQL, ALTER TYPE ADD VALUE no se puede ejecutar
    # dentro de una transacción. Usamos COMMIT implícito.
    # El valor se añade ANTES de 'uso_humano_no_financiado' para mantener orden lógico
    op.execute("""
        ALTER TYPE prescription_category_enum
        ADD VALUE IF NOT EXISTS 'no_comercializado'
        BEFORE 'uso_humano_no_financiado'
    """)
    print("✅ Valor 'no_comercializado' añadido al ENUM prescription_category_enum")

    # === VERIFICACIÓN FINAL ===
    value_added = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1 FROM pg_enum
            WHERE enumtypid = (SELECT oid FROM pg_type WHERE typname = 'prescription_category_enum')
              AND enumlabel = 'no_comercializado'
        )
    """)).scalar()

    if value_added:
        print("✅ Migración exitosa: valor 'no_comercializado' verificado")
    else:
        # Esto no debería pasar con IF NOT EXISTS, pero verificamos por seguridad
        print("⚠️  Advertencia: valor puede no haberse añadido correctamente")


def downgrade() -> None:
    """
    Remueve valor 'no_comercializado' del enum prescription_category_enum

    IMPORTANTE: PostgreSQL NO permite eliminar valores de un ENUM directamente.
    Para hacer un downgrade completo, necesitaríamos:
    1. Crear un nuevo ENUM sin el valor
    2. Actualizar la columna para usar el nuevo ENUM
    3. Eliminar el ENUM antiguo

    Por ahora, solo advertimos. Los valores huérfanos no causan problemas.
    """
    conn = op.get_bind()

    # Verificar si hay productos usando esta categoría
    products_using = conn.execute(sa.text("""
        SELECT COUNT(*)
        FROM product_catalog
        WHERE xfarma_prescription_category = 'no_comercializado'
    """)).scalar()

    if products_using and products_using > 0:
        print(f"⚠️  Hay {products_using} productos clasificados como NO_COMERCIALIZADO")
        print("    Estos productos se reclasificarán como USO_HUMANO_NO_FINANCIADO")

        # Reclasificar productos antes de intentar remover el valor
        op.execute("""
            UPDATE product_catalog
            SET xfarma_prescription_category = 'uso_humano_no_financiado'
            WHERE xfarma_prescription_category = 'no_comercializado'
        """)
        print(f"✅ {products_using} productos reclasificados")

    # NOTA: No podemos eliminar el valor del ENUM en PostgreSQL
    # sin recrear todo el tipo. Los valores huérfanos son inofensivos.
    print("⚠️  El valor 'no_comercializado' permanece en el ENUM (limitación PostgreSQL)")
    print("    Los productos han sido reclasificados a USO_HUMANO_NO_FINANCIADO")
