"""add_warnings_to_file_uploads

Issue #412: Agregar campo warnings (JSON) para avisos no-bloqueantes durante upload.

Revision ID: 20251126_warnings
Revises: dfb6fdfab99f
Create Date: 2025-11-26

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import JSON


# revision identifiers, used by Alembic.
revision: str = '20251126_warnings'
down_revision: Union[str, None] = 'dfb6fdfab99f'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Issue #412: Agregar campo warnings para separar avisos no-bloqueantes de errores fatales.

    El campo warnings es un JSON array que contiene objetos como:
    {
        "type": "encoding_fallback",
        "message": "Archivo detectado como LATIN1 (encoding típico de ERPs españoles)",
        "technical": "UTF-8 falló, se usó latin1"
    }
    """
    # REGLA #14 Alembic: Verificar existencia antes de modificar (idempotente)
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='file_uploads' AND column_name='warnings'"
    ))

    if not result.fetchone():
        # Column doesn't exist, add it
        op.add_column('file_uploads', sa.Column('warnings', JSON, nullable=True))
        # Inicializar con array vacío para registros existentes
        op.execute("UPDATE file_uploads SET warnings = '[]'::json WHERE warnings IS NULL")


def downgrade() -> None:
    """Issue #412: Remover campo warnings (idempotente - REGLA #14)."""
    # REGLA #14 Alembic: Verificar existencia antes de eliminar
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='file_uploads' AND column_name='warnings'"
    ))

    if result.fetchone():
        op.drop_column('file_uploads', 'warnings')
