"""Add catalog_sync_history table for Issue 349

Track synchronization history for CIMA and nomenclator catalog updates.
Provides visibility into sync operations for admin panel.

Issue #349: Admin panel database tab needs sync history viewer

Revision ID: 20251105_catalog_sync_history
Revises: fc38b8d391de
Create Date: 2025-11-05 06:20:50.000000+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision: str = '20251105_catalog_sync_history'
down_revision: Union[str, None] = 'fc38b8d391de'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Create catalog_sync_history table in idempotent way (REGLA #14).

    Table tracks each synchronization attempt with CIMA/nomenclator,
    including status, duration, records updated, and errors.
    """
    conn = op.get_bind()

    # ========================================================================
    # CRÍTICO: Verificar ENUMs FUERA del check de tabla (REGLA #14)
    # Razón: ENUMs pueden existir de migraciones previas fallidas o merges
    # Usar SQL directo para control total (no SQLAlchemy Enum.create())
    # ========================================================================

    # Check and create sync_type enum
    sync_type_exists = conn.execute(sa.text(
        "SELECT 1 FROM pg_type WHERE typname = 'synctype'"
    )).fetchone()

    if not sync_type_exists:
        conn.execute(sa.text("CREATE TYPE synctype AS ENUM ('cima', 'nomenclator')"))
        print("✅ Created synctype ENUM")
    else:
        print("ℹ️ synctype ENUM already exists, skipping creation")

    # Check and create sync_status enum
    sync_status_exists = conn.execute(sa.text(
        "SELECT 1 FROM pg_type WHERE typname = 'syncstatus'"
    )).fetchone()

    if not sync_status_exists:
        conn.execute(sa.text("CREATE TYPE syncstatus AS ENUM ('success', 'failure', 'partial')"))
        print("✅ Created syncstatus ENUM")
    else:
        print("ℹ️ syncstatus ENUM already exists, skipping creation")

    # Check and create trigger_type enum
    trigger_type_exists = conn.execute(sa.text(
        "SELECT 1 FROM pg_type WHERE typname = 'triggertype'"
    )).fetchone()

    if not trigger_type_exists:
        conn.execute(sa.text("CREATE TYPE triggertype AS ENUM ('automatic', 'manual')"))
        print("✅ Created triggertype ENUM")
    else:
        print("ℹ️ triggertype ENUM already exists, skipping creation")

    # ========================================================================
    # NOW check if table exists and create it
    # ========================================================================
    result = conn.execute(sa.text(
        "SELECT table_name FROM information_schema.tables "
        "WHERE table_schema='public' AND table_name='catalog_sync_history'"
    ))

    if not result.fetchone():

        # Create the table
        # CRÍTICO: Usar postgresql.ENUM con nombre de tipo existente (REGLA #14)
        # NO usar sa.Enum() porque siempre intenta crear el tipo incluso con create_type=False
        synctype_enum = postgresql.ENUM('cima', 'nomenclator', name='synctype', create_type=False)
        syncstatus_enum = postgresql.ENUM('success', 'failure', 'partial', name='syncstatus', create_type=False)
        triggertype_enum = postgresql.ENUM('automatic', 'manual', name='triggertype', create_type=False)

        op.create_table('catalog_sync_history',
            sa.Column('id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('sync_date', sa.DateTime(timezone=True), nullable=False),
            sa.Column('sync_type', synctype_enum, nullable=False),
            sa.Column('status', syncstatus_enum, nullable=False),
            sa.Column('records_updated', sa.Integer(), nullable=False),
            sa.Column('duration_seconds', sa.Float(), nullable=False),
            sa.Column('triggered_by', triggertype_enum, nullable=False),
            sa.Column('error_message', sa.Text(), nullable=True),
            sa.Column('created_at', sa.DateTime(timezone=True), nullable=False),
            sa.PrimaryKeyConstraint('id')
        )

        # Create indexes for performance (check if they don't exist first)
        # Index on sync_date for ordering
        sync_date_idx_exists = conn.execute(sa.text(
            "SELECT indexname FROM pg_indexes "
            "WHERE tablename = 'catalog_sync_history' "
            "AND indexname = 'idx_catalog_sync_history_sync_date'"
        )).fetchone()

        if not sync_date_idx_exists:
            op.create_index(
                'idx_catalog_sync_history_sync_date',
                'catalog_sync_history',
                ['sync_date'],
                unique=False
            )

        # Index on sync_type for filtering
        sync_type_idx_exists = conn.execute(sa.text(
            "SELECT indexname FROM pg_indexes "
            "WHERE tablename = 'catalog_sync_history' "
            "AND indexname = 'idx_catalog_sync_history_sync_type'"
        )).fetchone()

        if not sync_type_idx_exists:
            op.create_index(
                'idx_catalog_sync_history_sync_type',
                'catalog_sync_history',
                ['sync_type'],
                unique=False
            )

        # Index on status for filtering
        status_idx_exists = conn.execute(sa.text(
            "SELECT indexname FROM pg_indexes "
            "WHERE tablename = 'catalog_sync_history' "
            "AND indexname = 'idx_catalog_sync_history_status'"
        )).fetchone()

        if not status_idx_exists:
            op.create_index(
                'idx_catalog_sync_history_status',
                'catalog_sync_history',
                ['status'],
                unique=False
            )

        print("✅ Created catalog_sync_history table with indexes")
    else:
        print("ℹ️ Table catalog_sync_history already exists, skipping creation")


def downgrade() -> None:
    """
    Drop catalog_sync_history table and its associated enums in idempotent way.
    """
    # Check if table exists before dropping
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT table_name FROM information_schema.tables "
        "WHERE table_schema='public' AND table_name='catalog_sync_history'"
    ))

    if result.fetchone():
        # Drop indexes first
        op.drop_index('idx_catalog_sync_history_status', table_name='catalog_sync_history')
        op.drop_index('idx_catalog_sync_history_sync_type', table_name='catalog_sync_history')
        op.drop_index('idx_catalog_sync_history_sync_date', table_name='catalog_sync_history')

        # Drop the table
        op.drop_table('catalog_sync_history')

        # Drop enum types (check if they're not used by other tables)
        # Note: In production, you might want to keep enums if they could be reused
        conn.execute(sa.text("DROP TYPE IF EXISTS synctype"))
        conn.execute(sa.text("DROP TYPE IF EXISTS syncstatus"))
        conn.execute(sa.text("DROP TYPE IF EXISTS triggertype"))

        print("✅ Dropped catalog_sync_history table and associated types")
