"""add_performance_indexes_plan_standard

Optimización crítica para Plan Standard con 2 workers.
Índices para callbacks pesados identificados en análisis de rendimiento.

Performance Issues:
- Generic opportunities callback: 167s → Target: 40-60s (-60%)
- Partner analysis: 171s → Target: 50-70s (-58%)
- JOINs masivos sin índices en sales_data y enrichment tables

Impacto esperado:
- Queries 5-6x más rápidas (60s → 10-15s)
- Elimina full table scans en 50k+ registros
- Capacidad para 30-40 usuarios concurrentes

Revision ID: 2025110401
Revises: fc38b8d391de
Create Date: 2025-11-04 18:50:00.000000

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '2025110401'
down_revision: Union[str, None] = 'fc38b8d391de'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crea 4 índices críticos para optimizar callbacks pesados.

    Índices:
    1. idx_sales_pharmacy_date_product: Optimiza partner analysis y generic opportunities
    2. idx_enrichment_catalog_sales: Optimiza JOINs enrichment → catalog
    3. idx_sales_enrichment_lookup: Optimiza búsquedas enrichment por sales_data_id
    4. idx_product_catalog_code: Optimiza lookups de productos por código

    IMPORTANTE: Idempotente - verifica existencia antes de crear.
    """
    conn = op.get_bind()

    # ========================================================================
    # ÍNDICE 1: sales_data (pharmacy_id, sale_date DESC, codigo_nacional)
    # Optimiza: Partner analysis temporal, generic opportunities por período
    # ========================================================================
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_data' "
        "AND indexname = 'idx_sales_pharmacy_date_product'"
    ))

    if not result.fetchone():
        # Índice compuesto con sort descendente en sale_date
        # Note: Alembic no soporta DESC en create_index, pero PostgreSQL optimizará igual
        # CRÍTICO: Columna es 'codigo_nacional' NO 'product_code'
        op.create_index(
            'idx_sales_pharmacy_date_product',
            'sales_data',
            ['pharmacy_id', 'sale_date', 'codigo_nacional'],
            unique=False
        )
        print("[PERF] ✅ Created idx_sales_pharmacy_date_product")
    else:
        print("[PERF] ⏭️  idx_sales_pharmacy_date_product already exists")

    # ========================================================================
    # ÍNDICE 2: sales_enrichment (product_catalog_id, sales_data_id)
    # Optimiza: JOINs masivos enrichment → catalog en análisis
    # ========================================================================
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_enrichment' "
        "AND indexname = 'idx_enrichment_catalog_sales'"
    ))

    if not result.fetchone():
        op.create_index(
            'idx_enrichment_catalog_sales',
            'sales_enrichment',
            ['product_catalog_id', 'sales_data_id'],
            unique=False
        )
        print("[PERF] ✅ Created idx_enrichment_catalog_sales")
    else:
        print("[PERF] ⏭️  idx_enrichment_catalog_sales already exists")

    # ========================================================================
    # ÍNDICE 3: sales_enrichment (sales_data_id)
    # Optimiza: Lookups de enrichment data por venta
    # ========================================================================
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_enrichment' "
        "AND indexname = 'idx_sales_enrichment_lookup'"
    ))

    if not result.fetchone():
        op.create_index(
            'idx_sales_enrichment_lookup',
            'sales_enrichment',
            ['sales_data_id'],
            unique=False
        )
        print("[PERF] ✅ Created idx_sales_enrichment_lookup")
    else:
        print("[PERF] ⏭️  idx_sales_enrichment_lookup already exists")

    # ========================================================================
    # ÍNDICE 4: product_catalog (national_code)
    # Optimiza: Lookups de productos por código en enrichment
    # CRÍTICO: Columna es 'national_code' NO 'product_code'
    # ========================================================================
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'product_catalog' "
        "AND indexname = 'idx_product_catalog_code'"
    ))

    if not result.fetchone():
        op.create_index(
            'idx_product_catalog_code',
            'product_catalog',
            ['national_code'],
            unique=False
        )
        print("[PERF] ✅ Created idx_product_catalog_code")
    else:
        print("[PERF] ⏭️  idx_product_catalog_code already exists")


def downgrade() -> None:
    """
    Elimina los índices de forma segura e idempotente.
    """
    conn = op.get_bind()

    indexes = [
        ('sales_data', 'idx_sales_pharmacy_date_product'),
        ('sales_enrichment', 'idx_enrichment_catalog_sales'),
        ('sales_enrichment', 'idx_sales_enrichment_lookup'),
        ('product_catalog', 'idx_product_catalog_code'),
    ]

    for table_name, index_name in indexes:
        result = conn.execute(sa.text(
            f"SELECT indexname FROM pg_indexes "
            f"WHERE tablename = '{table_name}' "
            f"AND indexname = '{index_name}'"
        ))

        if result.fetchone():
            op.drop_index(index_name, table_name=table_name)
            print(f"[PERF] 🗑️  Dropped {index_name}")
        else:
            print(f"[PERF] ⏭️  {index_name} already dropped")
