"""fix pharmacy_partners UTF-8 encoding

Revision ID: fix_utf8_partners
Revises: (previous revision)
Create Date: 2025-11-01

Corrige encoding UTF-8 corrupto en pharmacy_partners.laboratory_name
Issue: Nombres con "ESPAÑA" aparecían como "ESPAÃA" (Ã en lugar de Ñ)

Registros afectados:
- TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A.
- MEDOCHEMIE IBERIA S.A. SUCURSAL EN ESPAÑA
- GRINDEKS KALCEKS ESPAÑA, S.L.
- FRESENIUS KABI ESPAÑA S.A.
- MACLEODS PHARMA ESPAÑA S.L.U.

Total: 10 registros (5 laboratorios × 2 registros cada uno)
"""
from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = '20251101_01_fix_utf8'
down_revision = '20251031_01_admin_panel'
branch_labels = None
depends_on = None


def upgrade():
    """
    Corrige encoding UTF-8 corrupto en nombres de laboratorios
    """
    # Usar raw SQL para máxima compatibilidad
    conn = op.get_bind()

    # MEDOCHEMIE
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = 'MEDOCHEMIE IBERIA S.A. SUCURSAL EN ESPAÑA',
            updated_at = NOW()
        WHERE laboratory_name LIKE '%MEDOCHEMIE%ESPAÃ%'
    """))

    # TECNIMEDE
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = 'TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A.',
            updated_at = NOW()
        WHERE laboratory_name LIKE '%TECNIMEDE%ESPAÃ%'
    """))

    # GRINDEKS KALCEKS
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = 'GRINDEKS KALCEKS ESPAÑA, S.L.',
            updated_at = NOW()
        WHERE laboratory_name LIKE '%GRINDEKS%ESPAÃ%'
    """))

    # FRESENIUS KABI
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = 'FRESENIUS KABI ESPAÑA S.A.',
            updated_at = NOW()
        WHERE laboratory_name LIKE '%FRESENIUS%ESPAÃ%'
    """))

    # MACLEODS
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = 'MACLEODS PHARMA ESPAÑA S.L.U.',
            updated_at = NOW()
        WHERE laboratory_name LIKE '%MACLEODS%ESPAÃ%'
    """))

    # Verificar corrección
    result = conn.execute(sa.text("""
        SELECT COUNT(*) as corrupted
        FROM pharmacy_partners
        WHERE laboratory_name LIKE '%Ã%'
    """))

    corrupted_count = result.scalar()
    if corrupted_count > 0:
        raise Exception(f"Still {corrupted_count} corrupted records after migration!")

    print("✅ UTF-8 encoding fixed for pharmacy_partners")


def downgrade():
    """
    Revertir corrección (restaurar corrupción original)

    NOTA: Esta operación NO es recomendada, solo se incluye por
    completitud del patrón Alembic.
    """
    conn = op.get_bind()

    # Restaurar estado corrupto original (solo para rollback de emergencia)
    conn.execute(sa.text("""
        UPDATE pharmacy_partners
        SET laboratory_name = REPLACE(laboratory_name, 'ESPAÑA', 'ESPAÃ'),
            updated_at = NOW()
        WHERE laboratory_name LIKE '%ESPAÑA%'
          AND (
              laboratory_name LIKE '%MEDOCHEMIE%'
              OR laboratory_name LIKE '%TECNIMEDE%'
              OR laboratory_name LIKE '%GRINDEKS%'
              OR laboratory_name LIKE '%FRESENIUS%'
              OR laboratory_name LIKE '%MACLEODS%'
          )
    """))

    print("⚠️ UTF-8 encoding reverted to corrupted state")
