"""admin_panel_rbac_subscriptions

Add RBAC (Role-Based Access Control) infrastructure, subscription plans,
storage tracking, and backup metadata for Admin Panel (Issue #348).

BACKGROUND:
This migration implements the database foundation for the Admin Panel Unificado
(Issue #348), enabling:
- RBAC with 6 granular permissions (MANAGE_USERS, DELETE_SALES_DATA, etc.)
- Subscription plans (free, pro, max) with storage limits
- Backup metadata tracking with integrity verification (SHA-256 + HMAC)
- Materialized view for pharmacy storage stats

PROBLEM:
- Current system uses simple is_admin boolean (insufficient for granular permissions)
- No subscription/storage management infrastructure
- No backup integrity verification
- Manual storage calculations slow with large datasets

SOLUTION:
- roles table: Centralized permission management (6 permissions × 2 roles)
- users.role_id: FK to roles (replaces is_admin)
- users.subscription_plan: 'free', 'pro', 'max' with CHECK constraint
- users.deleted_at: Soft delete for GDPR compliance
- pharmacy_storage_stats: Materialized view (auto-refresh for performance)
- backup_metadata: SHA-256 + HMAC signature verification
- 5 new indexes: Performance optimization for admin queries

BEHAVIORAL CHANGES:
Before:
- is_admin = True/False (binary, no granularity)
- No subscription limits
- Manual storage queries (slow)
- No backup integrity verification

After:
- RBAC with 6 permissions × 2 roles (admin, user)
- Storage tracking per pharmacy with subscription limits
- Materialized view refreshes every hour (CONCURRENTLY)
- Backup integrity verification with cryptographic signatures

Revision ID: 20251031_01_admin_panel
Revises: 20712ca991b2
Create Date: 2025-10-31 18:00:00.000000

"""
from typing import Sequence, Union
import json
import logging
import uuid

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# Configure logging
logger = logging.getLogger(__name__)

# revision identifiers, used by Alembic.
revision: str = '20251031_01_admin_panel'
down_revision: Union[str, None] = '20712ca991b2'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Add RBAC infrastructure, subscription plans, and admin panel features.

    Idempotent: Checks for existing tables/columns/indexes before creating.
    """
    conn = op.get_bind()

    # =================================================================
    # 1. CREATE TABLE: roles (RBAC foundation)
    # =================================================================
    logger.info("Checking if roles table exists...")
    result = conn.execute(sa.text(
        "SELECT table_name FROM information_schema.tables "
        "WHERE table_schema='public' AND table_name='roles'"
    ))

    if not result.fetchone():
        logger.info("Creating roles table...")
        op.create_table(
            'roles',
            sa.Column('id', postgresql.UUID(as_uuid=True), primary_key=True, default=uuid.uuid4),
            sa.Column('name', sa.String(50), unique=True, nullable=False),
            sa.Column('permissions', postgresql.JSONB, nullable=False),
            sa.Column('created_at', sa.DateTime(timezone=True), server_default=sa.text('NOW()')),
            sa.Column('updated_at', sa.DateTime(timezone=True), server_default=sa.text('NOW()'), onupdate=sa.text('NOW()'))
        )
        logger.info("✓ Created roles table")

        # Insert default roles (idempotent with ON CONFLICT)
        logger.info("Inserting default roles...")
        default_roles = [
            {
                "id": "22222222-2222-2222-2222-222222222222",
                "name": "admin",
                "permissions": ["manage_users", "delete_sales_data", "delete_accounts",
                              "manage_backups", "view_audit_logs", "clear_cache"]
            },
            {
                "id": "33333333-3333-3333-3333-333333333333",
                "name": "user",
                "permissions": []
            }
        ]

        for role in default_roles:
            permissions_json = json.dumps(role["permissions"])
            # Using sa.text() with manual string formatting (safe - no user input)
            query = sa.text(
                f"INSERT INTO roles (id, name, permissions) "
                f"VALUES ('{role['id']}', '{role['name']}', '{permissions_json}'::jsonb) "
                f"ON CONFLICT (name) DO NOTHING"
            )
            conn.execute(query)
        logger.info("✓ Inserted 2 default roles (admin, user)")
    else:
        logger.info("✓ roles table already exists (skipping)")

    # =================================================================
    # 2. ALTER TABLE users: Add role_id
    # =================================================================
    logger.info("Checking if users.role_id column exists...")
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='users' AND column_name='role_id'"
    ))

    if not result.fetchone():
        logger.info("Adding role_id column to users...")
        op.add_column(
            'users',
            sa.Column('role_id', postgresql.UUID(as_uuid=True), nullable=True)
        )

        # Set default role_id based on current role field
        logger.info("Migrating existing users to role_id...")
        conn.execute(sa.text(
            "UPDATE users SET role_id = "
            "(SELECT id FROM roles WHERE name = "
            "  CASE "
            "    WHEN users.role = 'admin' OR users.is_superuser = TRUE THEN 'admin' "
            "    ELSE 'user' "
            "  END"
            ")"
        ))

        # Set default value for new users (fallback to 'user' role)
        user_role_id = "33333333-3333-3333-3333-333333333333"  # 'user' role UUID
        conn.execute(sa.text(
            f"ALTER TABLE users ALTER COLUMN role_id SET DEFAULT '{user_role_id}'::uuid"
        ))

        # Make role_id NOT NULL after migration
        op.alter_column('users', 'role_id', nullable=False)

        # Add FK constraint
        op.create_foreign_key(
            'fk_users_role_id',
            'users', 'roles',
            ['role_id'], ['id'],
            ondelete='RESTRICT'  # Prevent role deletion if users exist
        )
        logger.info("✓ Added role_id column with FK to roles")
    else:
        logger.info("✓ users.role_id column already exists (skipping)")

    # =================================================================
    # 3. ALTER TABLE users: Add subscription_plan
    # =================================================================
    logger.info("Checking if users.subscription_plan column exists...")
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='users' AND column_name='subscription_plan'"
    ))

    if not result.fetchone():
        logger.info("Adding subscription_plan column to users...")
        op.add_column(
            'users',
            sa.Column(
                'subscription_plan',
                sa.String(50),
                nullable=False,
                server_default='free'
            )
        )

        # Add CHECK constraint for valid plans
        op.create_check_constraint(
            'check_subscription_plan',
            'users',
            "subscription_plan IN ('free', 'pro', 'max')"
        )
        logger.info("✓ Added subscription_plan column with CHECK constraint")
    else:
        logger.info("✓ users.subscription_plan column already exists (skipping)")

    # =================================================================
    # 4. ALTER TABLE users: Add deleted_at (soft delete)
    # =================================================================
    logger.info("Checking if users.deleted_at column exists...")
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='users' AND column_name='deleted_at'"
    ))

    if not result.fetchone():
        logger.info("Adding deleted_at column to users...")
        op.add_column(
            'users',
            sa.Column('deleted_at', sa.DateTime(timezone=True), nullable=True)
        )
        logger.info("✓ Added deleted_at column for soft delete")
    else:
        logger.info("✓ users.deleted_at column already exists (skipping)")

    # =================================================================
    # 5. CREATE INDEXES: Performance optimization for admin queries
    # =================================================================
    indexes = [
        ('idx_users_subscription_plan', 'users', ['subscription_plan']),
        ('idx_users_is_active', 'users', ['is_active']),
        ('idx_users_created_at_desc', 'users', ['created_at DESC']),
        ('idx_users_admin_filters', 'users', ['is_active', 'subscription_plan', 'created_at DESC'])
    ]

    for index_name, table_name, columns in indexes:
        logger.info(f"Checking if index {index_name} exists...")
        result = conn.execute(sa.text(
            f"SELECT indexname FROM pg_indexes WHERE indexname = '{index_name}'"
        ))

        if not result.fetchone():
            logger.info(f"Creating index {index_name}...")
            # Build column list for index
            columns_sql = ', '.join(columns)
            conn.execute(sa.text(
                f"CREATE INDEX {index_name} ON {table_name} ({columns_sql})"
            ))
            logger.info(f"✓ Created index {index_name}")
        else:
            logger.info(f"✓ Index {index_name} already exists (skipping)")

    # =================================================================
    # 6. CREATE INDEX: Partial index for soft delete
    # =================================================================
    logger.info("Checking if partial index idx_users_active exists...")
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes WHERE indexname = 'idx_users_active'"
    ))

    if not result.fetchone():
        logger.info("Creating partial index idx_users_active...")
        conn.execute(sa.text(
            "CREATE INDEX idx_users_active ON users(email) WHERE deleted_at IS NULL"
        ))
        logger.info("✓ Created partial index idx_users_active")
    else:
        logger.info("✓ Partial index idx_users_active already exists (skipping)")

    # =================================================================
    # 7. CREATE TABLE: backup_metadata (integrity tracking)
    # =================================================================
    logger.info("Checking if backup_metadata table exists...")
    result = conn.execute(sa.text(
        "SELECT table_name FROM information_schema.tables "
        "WHERE table_schema='public' AND table_name='backup_metadata'"
    ))

    if not result.fetchone():
        logger.info("Creating backup_metadata table...")
        op.create_table(
            'backup_metadata',
            sa.Column('id', postgresql.UUID(as_uuid=True), primary_key=True, default=uuid.uuid4),
            sa.Column('filename', sa.String(255), unique=True, nullable=False),
            sa.Column('sha256_hash', sa.String(64), nullable=False),
            sa.Column('hmac_signature', sa.String(64), nullable=False),
            sa.Column('size_bytes', sa.BigInteger, nullable=False),
            sa.Column('created_by_user_id', postgresql.UUID(as_uuid=True), nullable=True),
            sa.Column('created_at', sa.DateTime(timezone=True), server_default=sa.text('NOW()')),
            sa.ForeignKeyConstraint(['created_by_user_id'], ['users.id'], ondelete='SET NULL')
        )
        logger.info("✓ Created backup_metadata table")
    else:
        logger.info("✓ backup_metadata table already exists (skipping)")

    # =================================================================
    # 8. CREATE MATERIALIZED VIEW: pharmacy_storage_stats
    # =================================================================
    logger.info("Checking if materialized view pharmacy_storage_stats exists...")
    result = conn.execute(sa.text(
        "SELECT matviewname FROM pg_matviews WHERE matviewname = 'pharmacy_storage_stats'"
    ))

    if not result.fetchone():
        logger.info("Creating materialized view pharmacy_storage_stats...")
        conn.execute(sa.text("""
            CREATE MATERIALIZED VIEW pharmacy_storage_stats AS
            SELECT
                p.id as pharmacy_id,
                COUNT(DISTINCT sd.id) as total_sales,
                COALESCE(
                    (
                        COALESCE(SUM(pg_column_size(sd.*)), 0) +
                        COALESCE(SUM(pg_column_size(se.*)), 0) +
                        COALESCE(SUM(fu.file_size), 0)
                    ) / 1024 / 1024,
                    0
                ) as storage_mb,
                MIN(sd.created_at) as data_from,
                MAX(sd.created_at) as data_to
            FROM pharmacies p
            LEFT JOIN sales_data sd ON sd.pharmacy_id = p.id
            LEFT JOIN sales_enrichment se ON se.sales_data_id = sd.id
            LEFT JOIN file_uploads fu ON fu.pharmacy_id = p.id
            GROUP BY p.id
        """))

        # Create unique index for CONCURRENTLY refresh
        conn.execute(sa.text(
            "CREATE UNIQUE INDEX idx_pharmacy_storage_stats_pharmacy_id "
            "ON pharmacy_storage_stats(pharmacy_id)"
        ))
        logger.info("✓ Created materialized view pharmacy_storage_stats with unique index")
    else:
        logger.info("✓ Materialized view pharmacy_storage_stats already exists (skipping)")

    logger.info("Migration completed successfully!")


def downgrade() -> None:
    """
    Rollback RBAC infrastructure and admin panel features.

    WARNING: This will remove subscription plans and role assignments.
    """
    conn = op.get_bind()

    # Drop materialized view
    logger.info("Dropping materialized view pharmacy_storage_stats...")
    conn.execute(sa.text("DROP MATERIALIZED VIEW IF EXISTS pharmacy_storage_stats CASCADE"))

    # Drop backup_metadata table
    logger.info("Dropping backup_metadata table...")
    op.drop_table('backup_metadata')

    # Drop indexes
    indexes = [
        'idx_users_active',
        'idx_users_admin_filters',
        'idx_users_created_at_desc',
        'idx_users_is_active',
        'idx_users_subscription_plan'
    ]

    for index_name in indexes:
        logger.info(f"Dropping index {index_name}...")
        op.drop_index(index_name, table_name='users')

    # Drop columns from users
    logger.info("Dropping deleted_at column from users...")
    op.drop_column('users', 'deleted_at')

    logger.info("Dropping subscription_plan column from users...")
    op.drop_constraint('check_subscription_plan', 'users', type_='check')
    op.drop_column('users', 'subscription_plan')

    logger.info("Dropping role_id column from users...")
    op.drop_constraint('fk_users_role_id', 'users', type_='foreignkey')
    op.drop_column('users', 'role_id')

    # Drop roles table
    logger.info("Dropping roles table...")
    op.drop_table('roles')

    logger.info("Rollback completed!")
