"""add_user_configured_pharmacy_partners

Add user_configured field to pharmacy_partners table to persist user configuration
state across container restarts.

BACKGROUND:
This migration addresses Issue #PARTNERS_PERSISTENCE - Partners reset after Render restarts.

PROBLEM:
- User configures partners → Saved in DB
- Render restarts container (every 2-4h for scaling/memory)
- ensure_partners_initialized() detects "no user modifications"
- RE-INITIALIZES all partners → User configuration LOST

ROOT CAUSE:
- System relied on user_modified_at timestamp which could be NULL
- After restart, logic couldn't distinguish between:
  * Fresh pharmacy (needs auto-init)
  * Configured pharmacy (preserve user settings)

SOLUTION:
- Add user_configured BOOLEAN field (default: FALSE)
- Set to TRUE when user manually configures partners (PUT endpoint)
- NEVER re-initialize when user_configured=TRUE
- Field persists across restarts (unlike in-memory state)

BEHAVIORAL CHANGES:
Before:
- Auto-init if user_modified_at IS NULL OR selected_count=0
- User config lost after restart if modified_count couldn't be determined

After:
- Auto-init ONLY if user_configured=FALSE AND selected_count=0
- User config PERSISTS forever once user_configured=TRUE
- Explicit, unambiguous state tracking

Revision ID: 20251027_01_user_configured
Revises: df009b59a1e6
Create Date: 2025-10-27 12:00:00.000000

"""
from typing import Sequence, Union
import logging

from alembic import op
import sqlalchemy as sa

# Configure logging
logger = logging.getLogger(__name__)

# revision identifiers, used by Alembic.
revision: str = '20251027_01_user_configured'
down_revision: Union[str, None] = 'df009b59a1e6'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Add user_configured BOOLEAN field to pharmacy_partners table.

    Field Definition:
    - Type: BOOLEAN NOT NULL DEFAULT FALSE
    - Purpose: Track if user has manually configured partners
    - Behavior: Once TRUE, NEVER auto-reinitialize

    Idempotent: Checks for existing column before adding.
    """
    conn = op.get_bind()

    logger.info("Checking if user_configured column exists in pharmacy_partners...")
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='pharmacy_partners' AND column_name='user_configured'"
    ))

    if not result.fetchone():
        logger.info("Adding user_configured column to pharmacy_partners...")
        op.add_column(
            'pharmacy_partners',
            sa.Column(
                'user_configured',
                sa.Boolean(),
                nullable=False,
                server_default=sa.false(),
                comment='TRUE if user has manually configured partners (persists across restarts)'
            )
        )
        logger.info("✓ Added user_configured column with default FALSE")

        # Mark existing partners with user_modified_at as user_configured=TRUE
        logger.info("Marking existing user-modified partners as user_configured=TRUE...")
        conn.execute(sa.text(
            "UPDATE pharmacy_partners "
            "SET user_configured = TRUE "
            "WHERE user_modified_at IS NOT NULL"
        ))
        result = conn.execute(sa.text(
            "SELECT COUNT(*) FROM pharmacy_partners WHERE user_configured = TRUE"
        ))
        count = result.scalar()
        logger.info(f"✓ Marked {count} existing user-modified partners as configured")

    else:
        logger.info("Column user_configured already exists in pharmacy_partners, skipping")

    logger.info(
        "Successfully added/verified user_configured column. "
        "Partners will now persist across container restarts."
    )


def downgrade() -> None:
    """
    Remove user_configured column from pharmacy_partners table.

    WARNING: This will revert to old behavior where partners may be
    re-initialized after container restarts.

    Idempotent: Checks for column existence before dropping.
    """
    conn = op.get_bind()

    logger.warning(
        "DOWNGRADE: Removing user_configured column from pharmacy_partners. "
        "This will revert to old behavior where partners may reset after restarts!"
    )

    # Check if column exists before dropping
    result = conn.execute(sa.text(
        "SELECT column_name FROM information_schema.columns "
        "WHERE table_name='pharmacy_partners' AND column_name='user_configured'"
    ))

    if result.fetchone():
        logger.warning("Dropping user_configured column...")
        op.drop_column('pharmacy_partners', 'user_configured')
        logger.warning("✓ Dropped user_configured column")
    else:
        logger.info("Column user_configured does not exist, nothing to drop")

    logger.warning(
        "Downgrade complete. Partners may now be re-initialized after restarts. "
        "Consider re-upgrading to restore persistence."
    )
