"""add_indexes_file_uploads_performance

Add performance indexes for file_uploads table to optimize zombie cleanup queries.

This migration addresses performance concerns identified in zombie cleanup endpoint
(commit 0f1dacb) which queries file_uploads by:
- status (WHERE status = 'processing')
- processing_started_at (WHERE processing_started_at < cutoff_time)
- pharmacy_id (WHERE pharmacy_id = user_pharmacy_id)

PERFORMANCE IMPACT:
- Without indexes: O(n) full table scan for each query
- With indexes: O(log n) index scan
- Critical for production with 1000+ zombie uploads
- Transforms multi-second queries → < 100ms

INDEXES ADDED:
1. idx_file_uploads_status - Single field index on status
2. idx_file_uploads_processing_started_at - Single field index on processing_started_at
3. idx_file_uploads_pharmacy_status - Composite index on (pharmacy_id, status)

INDEX STRATEGY:
- Single field indexes: Good for admin queries (all pharmacies)
- Composite index: Optimal for non-admin queries (filtered by pharmacy_id + status)
- PostgreSQL query planner will choose best index automatically

Revision ID: 20251026_01_indexes
Revises: 826a662bc5da
Create Date: 2025-10-26 10:00:00.000000

"""
from typing import Sequence, Union
import logging

from alembic import op
import sqlalchemy as sa

# Configure logging
logger = logging.getLogger(__name__)

# revision identifiers, used by Alembic.
revision: str = '20251026_01_indexes'
down_revision: Union[str, None] = '826a662bc5da'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Add performance indexes on file_uploads table.

    Indexes:
    1. status - for zombie cleanup WHERE status = 'processing'
    2. processing_started_at - for time-based filtering
    3. (pharmacy_id, status) - composite for non-admin queries

    Idempotent: Checks for existing indexes before creating.
    """
    conn = op.get_bind()

    # Index 1: status (for all zombie cleanup queries)
    logger.info("Checking for index idx_file_uploads_status...")
    result = conn.execute(sa.text(
        "SELECT 1 FROM pg_indexes "
        "WHERE tablename='file_uploads' AND indexname='idx_file_uploads_status'"
    ))

    if not result.fetchone():
        logger.info("Creating index idx_file_uploads_status...")
        op.create_index(
            'idx_file_uploads_status',
            'file_uploads',
            ['status'],
            unique=False
        )
        logger.info("✓ Created index idx_file_uploads_status")
    else:
        logger.info("Index idx_file_uploads_status already exists, skipping")

    # Index 2: processing_started_at (for time-based filtering)
    logger.info("Checking for index idx_file_uploads_processing_started_at...")
    result = conn.execute(sa.text(
        "SELECT 1 FROM pg_indexes "
        "WHERE tablename='file_uploads' AND indexname='idx_file_uploads_processing_started_at'"
    ))

    if not result.fetchone():
        logger.info("Creating index idx_file_uploads_processing_started_at...")
        op.create_index(
            'idx_file_uploads_processing_started_at',
            'file_uploads',
            ['processing_started_at'],
            unique=False
        )
        logger.info("✓ Created index idx_file_uploads_processing_started_at")
    else:
        logger.info("Index idx_file_uploads_processing_started_at already exists, skipping")

    # Index 3: (pharmacy_id, status) composite (for non-admin queries)
    logger.info("Checking for index idx_file_uploads_pharmacy_status...")
    result = conn.execute(sa.text(
        "SELECT 1 FROM pg_indexes "
        "WHERE tablename='file_uploads' AND indexname='idx_file_uploads_pharmacy_status'"
    ))

    if not result.fetchone():
        logger.info("Creating composite index idx_file_uploads_pharmacy_status...")
        op.create_index(
            'idx_file_uploads_pharmacy_status',
            'file_uploads',
            ['pharmacy_id', 'status'],
            unique=False
        )
        logger.info("✓ Created composite index idx_file_uploads_pharmacy_status")
    else:
        logger.info("Index idx_file_uploads_pharmacy_status already exists, skipping")

    logger.info(
        "Successfully created/verified all performance indexes for file_uploads. "
        "Zombie cleanup queries should now be significantly faster."
    )


def downgrade() -> None:
    """
    Remove performance indexes from file_uploads table.

    WARNING: This will degrade performance of zombie cleanup queries.
    Only use for emergency rollback.
    """
    logger.warning(
        "DOWNGRADE: Removing performance indexes from file_uploads. "
        "This will degrade query performance!"
    )

    # Drop indexes in reverse order
    op.drop_index('idx_file_uploads_pharmacy_status', table_name='file_uploads')
    op.drop_index('idx_file_uploads_processing_started_at', table_name='file_uploads')
    op.drop_index('idx_file_uploads_status', table_name='file_uploads')

    logger.warning(
        "Performance indexes removed. Zombie cleanup queries will be slower. "
        "Consider re-upgrading if performance is critical."
    )
