"""add_sale_time_fix_duplicates

Add sale_time column and fix unique constraint to prevent legitimate duplicate rejections (fixes Issue #273)

This migration:
1. Adds sale_time column to sales_data table
2. Populates sale_time from existing sale_date timestamps
3. Drops old constraint that rejected legitimate sales
4. Creates new constraint including sale_time for proper duplicate detection

CONTEXT:
- Farmanager ERP includes timestamp with hour/minute/second
- Multiple sales of same product at same timestamp are LEGITIMATE (composite tickets)
- Old constraint: pharmacy_id + sale_date + codigo_nacional + product_name + quantity + total_amount
- New constraint: pharmacy_id + sale_date + sale_time + codigo_nacional + product_name + quantity + total_amount

SAFETY: This migration preserves all existing data and backfills sale_time

Revision ID: 20250114_sale_time
Revises: 20250112_unique_sales
Create Date: 2025-10-14 16:00:00.000000

"""
from typing import Sequence, Union
import logging

from alembic import op
from sqlalchemy import text

# Configure logging
logger = logging.getLogger(__name__)

# revision identifiers, used by Alembic.
revision: str = '20250114_sale_time'
down_revision: Union[str, None] = '4af31b1c94f6'  # add_composite_index_enrichment_progress_tracking
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Add sale_time column and update unique constraint to include time granularity.

    This fixes Issue #273: Legitimate sales with same product/amount at different times
    were being rejected as duplicates.
    """
    conn = op.get_bind()

    # STEP 1: Add sale_time column (nullable for existing data) - Skip if exists
    logger.info("Checking if sale_time column exists...")
    conn.execute(text("""
        DO $$
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM information_schema.columns
                WHERE table_name = 'sales_data' AND column_name = 'sale_time'
            ) THEN
                ALTER TABLE sales_data ADD COLUMN sale_time TIME;
            ELSE
                RAISE NOTICE 'Column sale_time already exists, skipping creation';
            END IF;
        END $$;
    """))

    # STEP 2: Backfill sale_time from existing sale_date timestamps
    logger.info("Backfilling sale_time from existing sale_date values...")
    conn.execute(text("""
        UPDATE sales_data
        SET sale_time = sale_date::time
        WHERE sale_time IS NULL
    """))

    rows_updated = conn.execute(text("SELECT COUNT(*) FROM sales_data WHERE sale_time IS NOT NULL")).fetchone()[0]
    null_rows = conn.execute(text("SELECT COUNT(*) FROM sales_data WHERE sale_time IS NULL")).fetchone()[0]
    logger.info(f"Backfilled sale_time for {rows_updated} existing records")
    if null_rows > 0:
        logger.warning(f"⚠️ {null_rows} records remain with NULL sale_time (expected for legacy data)")

    # STEP 3: Add index on sale_time for query performance - Skip if exists
    logger.info("Creating index on sale_time column...")
    conn.execute(text("""
        DO $$
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM pg_indexes
                WHERE indexname = 'ix_sales_data_sale_time'
            ) THEN
                CREATE INDEX ix_sales_data_sale_time ON sales_data (sale_time);
            ELSE
                RAISE NOTICE 'Index ix_sales_data_sale_time already exists, skipping creation';
            END IF;
        END $$;
    """))

    # STEP 4: Drop old constraint (without sale_time) - Skip if doesn't exist
    logger.info("Dropping old unique constraint 'uq_sales_data_composite'...")
    conn.execute(text("""
        DO $$
        BEGIN
            IF EXISTS (
                SELECT 1 FROM pg_constraint
                WHERE conname = 'uq_sales_data_composite'
            ) THEN
                ALTER TABLE sales_data DROP CONSTRAINT uq_sales_data_composite;
            ELSE
                RAISE NOTICE 'Constraint uq_sales_data_composite does not exist, skipping drop';
            END IF;
        END $$;
    """))

    # STEP 5: Create new constraint WITH sale_time - Skip if exists
    logger.info("Creating new unique constraint with sale_time included...")
    conn.execute(text("""
        DO $$
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM pg_constraint
                WHERE conname = 'uq_sales_data_composite_v2'
            ) THEN
                ALTER TABLE sales_data
                ADD CONSTRAINT uq_sales_data_composite_v2
                UNIQUE (pharmacy_id, sale_date, sale_time, codigo_nacional, product_name, quantity, total_amount);
            ELSE
                RAISE NOTICE 'Constraint uq_sales_data_composite_v2 already exists, skipping creation';
            END IF;
        END $$;
    """))

    logger.info(
        "✅ Migration completed successfully. "
        "Unique constraint now includes sale_time to allow legitimate sales at different times."
    )


def downgrade() -> None:
    """
    Revert changes: Remove sale_time column and restore old constraint.

    ⚠️  WARNING: This downgrade will allow false duplicates again!
    """
    logger.warning(
        "⚠️  DOWNGRADE: Removing sale_time column and reverting to old constraint. "
        "This will re-introduce Issue #273 (legitimate sales rejected as duplicates)."
    )

    # Drop new constraint
    logger.info("Dropping new constraint 'uq_sales_data_composite_v2'...")
    op.drop_constraint('uq_sales_data_composite_v2', 'sales_data', type_='unique')

    # Restore old constraint (without sale_time)
    logger.info("Restoring old constraint 'uq_sales_data_composite'...")
    op.create_unique_constraint(
        'uq_sales_data_composite',
        'sales_data',
        [
            'pharmacy_id',
            'sale_date',
            'codigo_nacional',
            'product_name',
            'quantity',
            'total_amount'
        ]
    )

    # Drop index
    logger.info("Dropping sale_time index...")
    op.drop_index('ix_sales_data_sale_time', 'sales_data')

    # Drop sale_time column
    logger.info("Dropping sale_time column...")
    op.drop_column('sales_data', 'sale_time')

    logger.warning(
        "⚠️  Downgrade completed. Issue #273 will re-occur. "
        "Please upgrade again as soon as possible."
    )
