"""add_prescription_analytics_indexes

Issue #400 Sprint 1: Índices adicionales para optimizar queries de prescription analytics.

Índices creados:
1. idx_prescription_category - Índice en xfarma_prescription_category (product_catalog)
2. idx_sales_date_pharmacy - Índice compuesto en (pharmacy_id, sale_date) (sales_data)
3. idx_atc_code_prefix - Índice funcional en LEFT(cima_atc_code, 1) (product_catalog)

Revision ID: 1931d756e755
Revises: a88279eeafbf
Create Date: 2025-11-09 17:04:37.291657+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '1931d756e755'
down_revision: Union[str, None] = 'a88279eeafbf'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crear índices para optimizar queries de prescription analytics.

    REGLA #14 (Alembic Workflow): Migraciones idempotentes.
    Verifica existencia ANTES de crear cada índice.
    """
    conn = op.get_bind()

    # Índice 1: xfarma_prescription_category (product_catalog)
    # Filtra productos de prescripción vs OTC/parafarmacia
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename='product_catalog' AND indexname='idx_prescription_category'"
    ))
    if not result.fetchone():
        op.create_index(
            'idx_prescription_category',
            'product_catalog',
            ['xfarma_prescription_category'],
            postgresql_where=sa.text('xfarma_prescription_category IS NOT NULL')
        )
        print("[MIGRATION] Created index: idx_prescription_category")
    else:
        print("[MIGRATION] Index already exists: idx_prescription_category")

    # Índice 2: (pharmacy_id, sale_date) compuesto (sales_data)
    # Optimiza filtros WHERE pharmacy_id = X AND sale_date BETWEEN Y AND Z
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename='sales_data' AND indexname='idx_sales_date_pharmacy'"
    ))
    if not result.fetchone():
        op.create_index(
            'idx_sales_date_pharmacy',
            'sales_data',
            ['pharmacy_id', 'sale_date']
        )
        print("[MIGRATION] Created index: idx_sales_date_pharmacy")
    else:
        print("[MIGRATION] Index already exists: idx_sales_date_pharmacy")

    # Índice 3: LEFT(cima_atc_code, 1) funcional (product_catalog)
    # Optimiza drill-down por niveles ATC (especialmente nivel 1 = primer carácter)
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename='product_catalog' AND indexname='idx_atc_code_prefix'"
    ))
    if not result.fetchone():
        # Índice funcional para primer carácter de ATC (nivel 1)
        op.execute(
            sa.text(
                "CREATE INDEX idx_atc_code_prefix "
                "ON product_catalog (LEFT(cima_atc_code, 1)) "
                "WHERE cima_atc_code IS NOT NULL"
            )
        )
        print("[MIGRATION] Created functional index: idx_atc_code_prefix")
    else:
        print("[MIGRATION] Index already exists: idx_atc_code_prefix")


def downgrade() -> None:
    """
    Revertir creación de índices.

    Safe to run múltiples veces (IF EXISTS).
    """
    # Drop índices en orden inverso
    op.execute(sa.text("DROP INDEX IF EXISTS idx_atc_code_prefix"))
    print("[MIGRATION] Dropped index: idx_atc_code_prefix")

    op.drop_index('idx_sales_date_pharmacy', table_name='sales_data', if_exists=True)
    print("[MIGRATION] Dropped index: idx_sales_date_pharmacy")

    op.drop_index('idx_prescription_category', table_name='product_catalog', if_exists=True)
    print("[MIGRATION] Dropped index: idx_prescription_category")
