"""initial_schema

Revision ID: 000_initial_schema
Revises:
Create Date: 2025-10-04 22:40:00.000000+00:00

DESCRIPCIÓN:
Esta migración crea TODAS las tablas del sistema desde los modelos SQLAlchemy.
Es thread-safe para multi-worker en Render.

TABLAS CREADAS (20 tablas):
- users: Usuarios con autenticación OAuth/JWT
- pharmacies: Farmacias registradas
- sales_data: Datos de ventas del ERP
- sales_enrichment: Datos enriquecidos con CIMA/nomenclator
- product_catalog: Catálogo de ~67k productos CIMA (37 campos)
- nomenclator_local: Precios oficiales y genéricos del Ministerio
- pharmacy_partners: Análisis de laboratorios colaboradores
- file_uploads: Tracking de archivos ERP subidos
- invitations: Sistema de invitaciones para onboarding
- audit_logs: Registro de auditoría para compliance
- system_status: Estado y progreso de componentes del sistema
- system_health_metrics: Métricas de salud del sistema
- system_alerts: Alertas para administradores
- developer_logs: Logs técnicos detallados
- performance_snapshots: Snapshots de performance
- homogeneous_groups: Grupos homogéneos de productos
- homogeneous_group_masters: Datos maestros de grupos homogéneos
- pharmacy_homogeneous_metrics: Métricas de grupos por farmacia
- cima_stall_events: Eventos de stall en sincronización CIMA (Issue #114)
"""
from typing import Sequence, Union

from alembic import op

# revision identifiers, used by Alembic.
revision: str = '000_initial_schema'
down_revision: Union[str, None] = None
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None

# FIX Issue #7: Configurable timeouts para operaciones críticas
import os as _os
BACKUP_TIMEOUT_SECONDS = int(_os.getenv("BACKUP_TIMEOUT", "300"))  # 5 min default
USER_CONFIRM_TIMEOUT_SECONDS = int(_os.getenv("USER_CONFIRM_TIMEOUT", "30"))  # 30s default


def upgrade() -> None:
    """
    Create all tables from SQLAlchemy models.

    IMPORTANTE: Importamos TODOS los modelos para que Base.metadata
    tenga conocimiento de todas las tablas a crear.

    Thread-safety: Todos los modelos usan __table_args__ = {'extend_existing': True}
    para evitar conflictos en entornos multi-worker (Render).
    """
    # Importar TODOS los modelos desde app.models
    # Esto registra todas las tablas en Base.metadata
    from app.models import Base

    # Crear todas las tablas en la base de datos
    # Base.metadata contiene información de todas las tablas importadas arriba
    Base.metadata.create_all(bind=op.get_bind())


def downgrade() -> None:
    """
    Drop all tables.

    ⚠️  ADVERTENCIA CRÍTICA: Esta operación DESTRUIRÁ TODOS LOS DATOS ⚠️

    PROTECCIONES IMPLEMENTADAS (Issue #[INVESTIGACIÓN] - Pérdida de datos):
    1. PROHIBIDO en producción (auto-abort)
    2. Requiere confirmación explícita en desarrollo
    3. Crea backup automático ANTES de destruir
    4. Registra operación en audit logs

    En producción, NUNCA usar downgrade. Usar migraciones forward o contactar DBA.
    """
    import sys
    import os
    from datetime import datetime

    def get_user_confirmation_with_timeout(timeout: int = 30) -> str:
        """
        Obtener confirmación del usuario con timeout.

        FIX Issue #2: Usar threading para evitar busy-wait en Windows.

        Args:
            timeout: Segundos a esperar antes de timeout

        Returns:
            str: Respuesta del usuario o None si timeout o stdin no disponible
        """
        # Check si stdin está disponible (no en CI/CD)
        if not sys.stdin.isatty():
            print("  ⚠️  stdin no disponible (CI/CD?), asumiendo cancelación")
            return None

        try:
            # En Windows, usar threading para leer input sin busy-wait
            if sys.platform == 'win32':
                import threading

                result = [None]  # Mutable para compartir entre threads

                def read_input():
                    """Thread que lee input bloqueante"""
                    try:
                        result[0] = input()
                    except Exception as e:
                        print(f"\n  ⚠️  Error leyendo input: {e}")
                        result[0] = None

                # Crear y lanzar thread daemon
                read_thread = threading.Thread(target=read_input, daemon=True)
                read_thread.start()

                # Esperar con timeout
                read_thread.join(timeout=timeout)

                if read_thread.is_alive():
                    # Timeout alcanzado
                    print(f"\n  ⚠️  Timeout ({timeout}s) - Operación cancelada")
                    return None
                else:
                    # Thread terminó, retornar resultado
                    return result[0]

            else:
                # Unix/Linux: usar select (ya eficiente)
                import select
                ready, _, _ = select.select([sys.stdin], [], [], timeout)
                if ready:
                    return sys.stdin.readline().strip()
                else:
                    print(f"\n  ⚠️  Timeout ({timeout}s) - Operación cancelada")
                    return None

        except Exception as e:
            print(f"\n  ⚠️  Error leyendo input: {e}")
            return None

    # 🛑 PROTECCIÓN #1: PROHIBIR downgrade en producción
    environment = os.getenv("ENVIRONMENT", "development")
    if environment == "production":
        error_msg = (
            "\n" + "="*70 + "\n"
            "❌ DOWNGRADE PROHIBIDO EN PRODUCCIÓN\n"
            "="*70 + "\n"
            "Esta operación destruiría TODOS los datos de producción.\n\n"
            "Para revertir cambios en producción:\n"
            "  1. Crear migración forward que revierta cambios\n"
            "  2. Contactar al DBA para procedimiento de emergencia\n"
            "  3. Restaurar desde backup si es absolutamente necesario\n\n"
            "Documentación: docs/RECOVERY_PROCEDURES.md\n"
        )
        print(error_msg, file=sys.stderr)

        # Log crítico antes de abortar
        try:
            import structlog
            logger = structlog.get_logger()
            logger.critical(
                "migration.downgrade.blocked_production",
                environment=environment,
                revision="000_initial_schema",
                attempted_by=os.getenv("USER", "unknown")
            )
        except:
            pass

        sys.exit(1)

    # 🛑 PROTECCIÓN #2: REQUERIR confirmación explícita en desarrollo
    confirm = os.getenv("ALEMBIC_DOWNGRADE_CONFIRMED")
    if confirm != "YES_I_UNDERSTAND_DATA_WILL_BE_LOST":
        print("\n" + "="*70)
        print("⚠️  OPERACIÓN DESTRUCTIVA - SE ELIMINARÁN TODOS LOS DATOS")
        print("="*70)
        print("\nEsta operación:")
        print("  • Eliminará TODAS las tablas (20 tablas)")
        print("  • Perderás TODOS los datos almacenados")
        print("  • NO es reversible sin un backup")
        print("\n💡 Recomendación: Ejecuta backup manual primero:")
        print("     .\\scripts\\backup-db.ps1")
        print("\nPara confirmar, ejecuta:")
        print("  $env:ALEMBIC_DOWNGRADE_CONFIRMED='YES_I_UNDERSTAND_DATA_WILL_BE_LOST'")
        print("  alembic downgrade base")
        print()
        sys.exit(1)

    # 🛑 PROTECCIÓN #3: Crear backup automático ANTES de destruir
    try:
        import subprocess
        from pathlib import Path

        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_dir = Path("backups")
        backup_dir.mkdir(parents=True, exist_ok=True)
        backup_file = backup_dir / f"pre_downgrade_{timestamp}.sql"

        # Obtener credenciales desde variables de entorno (NO hardcodeadas)
        db_user = os.getenv("POSTGRES_USER", "xfarma_user")
        db_password = os.getenv("POSTGRES_PASSWORD", "xfarma_dev_2024")
        db_name = os.getenv("POSTGRES_DB", "xfarma_db")
        container_name = os.getenv("POSTGRES_CONTAINER", "xfarma_postgres")

        print(f"\n[1/3] Creando backup de seguridad en: {backup_file}")

        # Usar subprocess.run() con lista de argumentos (NO shell=True)
        # Esto previene command injection
        with open(backup_file, 'w') as f:
            result = subprocess.run(
                [
                    "docker", "exec", container_name,
                    "pg_dump",
                    "-U", db_user,
                    "-d", db_name,
                    "--clean", "--if-exists"
                ],
                stdout=f,
                stderr=subprocess.PIPE,
                env={**os.environ, "PGPASSWORD": db_password},
                timeout=BACKUP_TIMEOUT_SECONDS
            )

        # FIX Issue #1: TOCTOU - Usar try/except para operaciones atómicas
        backup_created = False
        try:
            if result.returncode == 0:
                # Operación atómica: stat() devolverá FileNotFoundError si desapareció
                file_size = backup_file.stat().st_size
                if file_size > 0:
                    file_size_mb = file_size / (1024 * 1024)
                    print(f"  ✅ Backup creado: {file_size_mb:.2f} MB")

                    # FIX Issue #3: Verificar integridad del backup
                    print("  🔍 Verificando integridad del backup...")

                    # Leer primeras líneas para verificar formato SQL válido
                    with open(backup_file, 'r', encoding='utf-8', errors='ignore') as f:
                        first_lines = [f.readline() for _ in range(10)]
                        content_preview = ''.join(first_lines).lower()

                        # Verificar que contiene comandos SQL típicos de pg_dump
                        required_keywords = ['postgresql', 'database', 'dump']
                        sql_keywords = ['create', 'insert', 'table', 'schema']

                        has_pg_dump_header = any(kw in content_preview for kw in required_keywords)
                        has_sql_content = any(kw in content_preview for kw in sql_keywords)

                        if has_pg_dump_header and has_sql_content:
                            print("  ✅ Backup verificado correctamente")
                            backup_created = True
                        else:
                            raise ValueError(
                                f"Backup no contiene formato SQL válido. "
                                f"PG_DUMP header: {has_pg_dump_header}, "
                                f"SQL keywords: {has_sql_content}"
                            )
                else:
                    raise ValueError("Backup file is empty")
        except (FileNotFoundError, ValueError, subprocess.TimeoutExpired) as e:
            result.returncode = 1  # Marcar como fallido para manejo uniforme
            print(f"  ⚠️  Error validando backup: {e}")

        if not backup_created:
            error_output = result.stderr.decode('utf-8') if result.stderr else "Unknown error"
            print("  ⚠️  Advertencia: No se pudo crear backup automático")
            print(f"  Error: {error_output[:200]}")
            print("  Continuar sin backup? (escribe 'CONTINUE_WITHOUT_BACKUP')")

            # Usar get_user_confirmation con timeout
            response = get_user_confirmation_with_timeout(timeout=USER_CONFIRM_TIMEOUT_SECONDS)
            if response != "CONTINUE_WITHOUT_BACKUP":
                print("  ❌ Operación cancelada - Backup no disponible")
                sys.exit(1)
            backup_created = False
            backup_file = None

    except subprocess.TimeoutExpired:
        print(f"  ❌ Error: Timeout creando backup (>{BACKUP_TIMEOUT_SECONDS}s)")
        print("  Continuar sin backup? (escribe 'CONTINUE_WITHOUT_BACKUP')")
        response = get_user_confirmation_with_timeout(timeout=USER_CONFIRM_TIMEOUT_SECONDS)
        if response != "CONTINUE_WITHOUT_BACKUP":
            sys.exit(1)
        backup_created = False
        backup_file = None
    except Exception as e:
        print(f"  ❌ Error creando backup: {e}")
        print("  Continuar sin backup? (escribe 'CONTINUE_WITHOUT_BACKUP')")
        response = get_user_confirmation_with_timeout(timeout=USER_CONFIRM_TIMEOUT_SECONDS)
        if response != "CONTINUE_WITHOUT_BACKUP":
            sys.exit(1)
        backup_created = False
        backup_file = None

    # 🛑 PROTECCIÓN #4: Registrar en audit logs ANTES de destruir
    try:
        from app.middleware.destructive_operations_logger import log_alembic_downgrade

        log_alembic_downgrade(
            revision="000_initial_schema",
            user=os.getenv("USER", "unknown"),
            confirmed=True
        )
        print("[2/3] Operación registrada en audit logs")

    except Exception as e:
        print(f"  ⚠️  Advertencia: No se pudo registrar en audit logs: {e}")

    # EJECUTAR DOWNGRADE (punto de no retorno)
    print("[3/3] Ejecutando drop tables (GRANULAR) - Eliminando tablas una a una...")

    # ⚠️ FIXED: Usar op.drop_table() granular en lugar de drop_all()
    # Esto permite downgrades parciales sin perder TODAS las tablas

    # ORDEN CRÍTICO: Eliminar tablas en orden inverso de dependencias (FK constraints)

    # 1. Tablas sin dependencias (pueden eliminarse primero)
    print("  Eliminando tablas de sistema y logs...")
    op.drop_table('cima_stall_events')
    op.drop_table('performance_snapshots')
    op.drop_table('developer_logs')
    op.drop_table('system_alerts')
    op.drop_table('system_health_metrics')
    op.drop_table('system_status')

    # 2. Tablas de auditoría y autenticación
    print("  Eliminando tablas de autenticación...")
    op.drop_table('audit_logs')
    op.drop_table('invitations')

    # 3. Tablas de analytics (dependen de sales/pharmacy)
    print("  Eliminando tablas de analytics...")
    op.drop_table('pharmacy_homogeneous_metrics')
    op.drop_table('homogeneous_groups')
    op.drop_table('homogeneous_groups_master')
    op.drop_table('pharmacy_partners')

    # 4. Tablas de sales (dependen de catalog y pharmacy)
    print("  Eliminando tablas de ventas...")
    op.drop_table('sales_enrichment')
    op.drop_table('sales_data')
    op.drop_table('file_uploads')

    # 5. Tablas de catálogo (sin dependencias de users/pharmacy)
    print("  Eliminando tablas de catálogo...")
    op.drop_table('nomenclator_local')
    op.drop_table('product_catalog')

    # 6. Tabla users (depende de pharmacy)
    print("  Eliminando tabla users...")
    op.drop_table('users')

    # 7. Tabla pharmacy (última, base de todo)
    print("  Eliminando tabla pharmacies...")
    op.drop_table('pharmacies')

    print("\n✅ Downgrade completado")
    if backup_created:
        print(f"📦 Backup disponible en: {backup_file}")
        print(f"   Para restaurar: .\\scripts\\restore-db.ps1 -BackupFile '{os.path.basename(backup_file)}'")
    print()
