"""
TEMPLATE SEGURO PARA MIGRACIONES XFARMA

Utiliza este template para crear migraciones robustas que previenen
errores comunes como duplicación de columnas/índices.

USO:
1. Copia este archivo y renombra
2. Reemplaza placeholders con datos reales
3. Utiliza las funciones safe_* importadas

EJEMPLO:
```python
# En lugar de:
op.add_column('table', sa.Column('col', sa.String(50)))

# Usa:
safe_add_column('table', sa.Column('col', sa.String(50)))
```
"""

from typing import Sequence, Union

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects import postgresql

# IMPORTAR SIEMPRE las utilidades de seguridad
from .migration_utils import (
    safe_add_column,
    safe_create_index,
    safe_create_table,
    safe_drop_column,
    safe_drop_index,
    safe_drop_table,
    table_exists,
    validate_migration_safety,
)

# revision identifiers, used by Alembic.
revision: str = "TEMPLATE_REVISION_ID"
down_revision: Union[str, None] = "PREVIOUS_REVISION_ID"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    TEMPLATE - Función de migración hacia adelante

    REGLAS OBLIGATORIAS:
    1. SIEMPRE usar funciones safe_* para operaciones DDL
    2. VALIDAR operaciones antes de ejecutar con validate_migration_safety
    3. IMPRIMIR información detallada del progreso
    4. MANEJAR errores explícitamente con try/except
    """

    print("=== INICIANDO MIGRACIÓN TEMPLATE ===")

    # VALIDACIÓN PREVENTIVA (recomendado para operaciones complejas)
    operations = [{"type": "add_column", "name": "COLUMN_NAME"}, {"type": "create_index", "name": "INDEX_NAME"}]
    warnings = validate_migration_safety("TABLE_NAME", operations)
    if warnings:
        print("⚠️  ADVERTENCIAS DE MIGRACIÓN:")
        for warning in warnings:
            print(f"   - {warning}")

    # OPERACIONES SEGURAS DE EJEMPLO
    try:
        # Agregar columna de forma segura
        new_column = sa.Column("new_column_name", sa.String(100), nullable=True)
        if safe_add_column("target_table", new_column):
            print("✅ Columna agregada exitosamente")

        # Crear índice de forma segura
        if safe_create_index("ix_target_table_new_column", "target_table", ["new_column_name"]):
            print("✅ Índice creado exitosamente")

        # Crear tabla de forma segura
        if safe_create_table(
            "new_table",
            sa.Column("id", postgresql.UUID(), primary_key=True),
            sa.Column("name", sa.String(100), nullable=False),
            sa.Column("created_at", sa.DateTime, server_default=sa.func.now()),
        ):
            print("✅ Tabla creada exitosamente")

    except Exception as e:
        print(f"❌ ERROR durante la migración: {e}")
        raise

    print("=== MIGRACIÓN TEMPLATE COMPLETADA ===")


def downgrade() -> None:
    """
    TEMPLATE - Función de rollback

    REGLAS OBLIGATORIAS:
    1. SIEMPRE implementar rollback completo
    2. ORDEN INVERSO: eliminar en orden opuesto a la creación
    3. USAR safe_* para operaciones de eliminación también
    """

    print("=== INICIANDO ROLLBACK TEMPLATE ===")

    try:
        # Rollback en orden inverso

        # 1. Eliminar tabla si existe
        if safe_drop_table("new_table"):
            print("✅ Tabla eliminada exitosamente")

        # 2. Eliminar índice si existe
        if safe_drop_index("ix_target_table_new_column", "target_table"):
            print("✅ Índice eliminado exitosamente")

        # 3. Eliminar columna si existe
        if safe_drop_column("target_table", "new_column_name"):
            print("✅ Columna eliminada exitosamente")

    except Exception as e:
        print(f"❌ ERROR durante el rollback: {e}")
        raise

    print("=== ROLLBACK TEMPLATE COMPLETADO ===")


# FUNCIONES AUXILIARES ESPECÍFICAS (opcional)
def custom_validation() -> bool:
    """
    Validación personalizada específica para esta migración
    """
    # Ejemplo: verificar integridad de datos antes de migrar
    connection = op.get_bind()
    result = connection.execute(sa.text("SELECT COUNT(*) FROM target_table WHERE condition = 'specific'"))
    count = result.scalar()

    if count > 1000:
        print(f"⚠️  Advertencia: {count} registros afectados por la migración")
        return True

    return True


def backup_critical_data():
    """
    Backup de datos críticos antes de modificaciones riesgosas
    """
    print("📦 Creando backup de datos críticos...")

    # Ejemplo: crear tabla temporal con datos importantes
    if not table_exists("backup_critical_data"):
        op.execute(
            """
        CREATE TABLE backup_critical_data AS
        SELECT id, critical_field, created_at
        FROM target_table
        WHERE critical_field IS NOT NULL
        """
        )
        print("✅ Backup creado exitosamente")
    else:
        print("ℹ️  Backup ya existe, omitiendo...")


def cleanup_backup_data():
    """
    Limpieza del backup después de verificar integridad
    """
    print("🧹 Limpiando datos de backup...")

    if table_exists("backup_critical_data"):
        op.execute("DROP TABLE backup_critical_data")
        print("✅ Backup eliminado exitosamente")


# PATRÓN PARA MIGRACIONES DE DATOS MASIVOS
def migrate_data_in_batches(batch_size: int = 1000):
    """
    Migrar datos en lotes para evitar timeouts en datasets grandes
    """
    connection = op.get_bind()

    # Contar total de registros
    total_count = connection.execute(sa.text("SELECT COUNT(*) FROM target_table")).scalar()

    print(f"📊 Migrando {total_count} registros en lotes de {batch_size}")

    offset = 0
    processed = 0

    while offset < total_count:
        # Procesar lote
        connection.execute(
            sa.text(
                """
            UPDATE target_table
            SET new_field = CASE
                WHEN old_field = 'A' THEN 'VALUE_A'
                WHEN old_field = 'B' THEN 'VALUE_B'
                ELSE 'DEFAULT'
            END
            WHERE id IN (
                SELECT id FROM target_table
                ORDER BY id
                LIMIT :batch_size OFFSET :offset
            )
            """
            ),
            {"batch_size": batch_size, "offset": offset},
        )

        offset += batch_size
        processed = min(offset, total_count)
        progress = (processed / total_count) * 100

        print(f"📈 Progreso: {processed}/{total_count} ({progress:.1f}%)")

    print("✅ Migración de datos completada")
